/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC_SingleConversion C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2021/09/03      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"
/* LiteOS 头文件 */
#include "los_sys.h"
#include "los_task.ph"
#include "los_swtmr.h"
/* 板级外设头文件 */
#include "debug_uart.h"
#include "debug_gpio.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */

static UINT32 AppTaskCreate(void);
static void Timer1_Callback(UINT32 arg);
static void Timer2_Callback(UINT32 arg);

static void LED_Task(void);
static void Key_Task(void);
static void BSP_Init(void);

/* Private Variables --------------------------------------------------------- */

static UINT32 s_TmrCb_Count1 = 0;
static UINT32 s_TmrCb_Count2 = 0;

/* Public Variables ---------------------------------------------------------- */

/* 定义定时器句柄（ID） */
UINT16 g_Timer1_Handle;
UINT16 g_Timer2_Handle;

/* Private Function ---------------------------------------------------------- */

/***************************************************************
  * @brief  主函数
  * @param  无
  * @retval 无
  * @note   第一步：开发板硬件初始化
            第二步：创建APP应用任务
            第三步：启动LiteOS，开始多任务调度，启动失败则输出错误信息
  **************************************************************/
int main(void)
{
    UINT32 uwRet = LOS_OK;  /* 定义一个任务创建的返回值，默认为创建成功 */

    /* 板载相关初始化 */
    BSP_Init();

    printf("example start\r\n");

    /* LiteOS 内核初始化 */
    uwRet = LOS_KernelInit();

    if (uwRet != LOS_OK)
    {
        printf("[err]LiteOS core init err! code = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    uwRet = AppTaskCreate();

    if (uwRet != LOS_OK)
    {
        printf("[err]AppTaskCreate err! cde = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    /* 开启LiteOS任务调度 */
    LOS_Start();

    /* 正常情况下不会执行到这里 */
    while (1);

}


/*********************************************************************
  * @ 函数名  ： AppTaskCreate
  * @ 功能说明： 任务创建，为了方便管理，所有的任务创建函数都可以放在这个函数里面
  * @ 参数    ： 无
  * @ 返回值  ： 无
  *******************************************************************/
static UINT32 AppTaskCreate(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 创建一个软件定时器*/
    uwRet = LOS_SwtmrCreate(5000,                   /* 软件定时器的定时时间*/
                            LOS_SWTMR_MODE_ONCE,    /* 软件定时器模式 一次模式 */
                            (SWTMR_PROC_FUNC)Timer1_Callback,       /* 软件定时器的回调函数 */
                            &g_Timer1_Handle,                   /* 软件定时器的id */
                            0);

    if (uwRet != LOS_OK)
    {
        printf("Timer1 create err! code = 0x%X\r\n", uwRet);
    }

    uwRet = LOS_SwtmrCreate(1000,                   /* 软件定时器的定时时间（ms）*/
                            LOS_SWTMR_MODE_PERIOD,  /* 软件定时器模式 周期模式 */
                            (SWTMR_PROC_FUNC)Timer2_Callback,       /* 软件定时器的回调函数 */
                            &g_Timer2_Handle,           /* 软件定时器的id */
                            0);

    if (uwRet != LOS_OK)
    {
        printf("Timer2 create err! code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    /* 启动一个软件定时器*/
    uwRet = LOS_SwtmrStart(g_Timer1_Handle);

    if (uwRet != LOS_OK)
    {
        printf("start Timer1 failed\r\n");
        return uwRet;
    }
    else
    {
        printf("start Timer1 sucess\r\n");
    }

    /* 启动一个软件定时器*/
    uwRet = LOS_SwtmrStart(g_Timer2_Handle);

    if (LOS_OK != uwRet)
    {
        printf("start Timer2 failed\r\n");
        return uwRet;
    }
    else
    {
        printf("start Timer2 succes\r\n");
    }

    return LOS_OK;
}

/*********************************************************************************
  * @ 函数名  ： Timer1_Callback
  * @ 功能说明： 软件定时器回调函数1
  * @ 参数    ： 传入1个参数，但未使用
  * @ 返回值  ： 无
  ********************************************************************************/
static void Timer1_Callback(UINT32 arg)
{
    UINT32 tick_num1;

    s_TmrCb_Count1++;                       /* 每回调一次加一 */
    DEBUG_LED1_TOGGLE;
    tick_num1 = (UINT32)LOS_TickCountGet(); /* 获取滴答定时器的计数值 */

    printf("Timer_CallBack_Count1=%d\r\n", s_TmrCb_Count1);
    printf("tick_num1=%d\r\n", tick_num1);
}
/*********************************************************************************
  * @ 函数名  ： Timer2_Callback
  * @ 功能说明： 软件定时器回调函数2
  * @ 参数    ： 传入1个参数，但未使用
  * @ 返回值  ： 无
  ********************************************************************************/
static void Timer2_Callback(UINT32 arg)
{
    UINT32 tick_num2;

    s_TmrCb_Count2++;               /* 每回调一次加一 */
    DEBUG_LED2_TOGGLE;
    tick_num2 = (UINT32)LOS_TickCountGet(); /* 获取滴答定时器的计数值 */

    printf("Timer_CallBack_Count2=%d\r\n", s_TmrCb_Count2);

    printf("tick_num2=%d\r\n", tick_num2);

}

/*******************************************************************
  * @ 函数名  ： BSP_Init
  * @ 功能说明： 板级外设初始化，所有板子上的初始化均可放在这个函数里面
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static void BSP_Init(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    debug_gpio_init();
    debug_uart_init();
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
