/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC_SingleConversion C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2021/09/03      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"
/* LiteOS 头文件 */
#include "los_sys.h"
#include "los_task.ph"
/* 板级外设头文件 */
#include "debug_uart.h"
#include "debug_gpio.h"

/* Private Macros ------------------------------------------------------------ */

#define KEY1_EVENT  (0x01 << 0)/* 设置事件掩码的位0 */
#define KEY2_EVENT  (0x01 << 1)/* 设置事件掩码的位1 */

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */

static UINT32 AppTaskCreate(void);
static UINT32 Creat_LED_Task(void);
static UINT32 Creat_Key_Task(void);

static void LED_Task(void);
static void Key_Task(void);
static void BSP_Init(void);

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* 定义任务句柄 */
UINT32 g_LED_Task_Handle;
UINT32 g_Key_Task_Handle;

/* 定义事件标志组的控制块 */
static EVENT_CB_S s_EventGroup_CB;

/* Private Function ---------------------------------------------------------- */

/***************************************************************
  * @brief  主函数
  * @param  无
  * @retval 无
  * @note   第一步：开发板硬件初始化
            第二步：创建APP应用任务
            第三步：启动LiteOS，开始多任务调度，启动失败则输出错误信息
  **************************************************************/
int main(void)
{
    UINT32 uwRet = LOS_OK;  /* 定义一个任务创建的返回值，默认为创建成功 */

    /* 板载相关初始化 */
    BSP_Init();

    printf("example start\r\n");

    /* LiteOS 内核初始化 */
    uwRet = LOS_KernelInit();

    if (uwRet != LOS_OK)
    {
        printf("[err]LiteOS core init err! code = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    uwRet = AppTaskCreate();

    if (uwRet != LOS_OK)
    {
        printf("[err]AppTaskCreate err! cde = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    /* 开启LiteOS任务调度 */
    LOS_Start();

    /* 正常情况下不会执行到这里 */
    while (1);

}


/*********************************************************************
  * @ 函数名  ： AppTaskCreate
  * @ 功能说明： 任务创建，为了方便管理，所有的任务创建函数都可以放在这个函数里面
  * @ 参数    ： 无
  * @ 返回值  ： 无
  *******************************************************************/
static UINT32 AppTaskCreate(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 创建一个事件标志组*/
    uwRet = LOS_EventInit(&s_EventGroup_CB);

    if (uwRet != LOS_OK)
    {
        printf("s_EventGroup_CB create err! code = 0x%X\r\n", uwRet);
    }

    uwRet = Creat_LED_Task();

    if (uwRet != LOS_OK)
    {
        printf("LED_Task create err! code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    uwRet = Creat_Key_Task();

    if (uwRet != LOS_OK)
    {
        printf("Key_Task create err! code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    return LOS_OK;
}


/******************************************************************
  * @ 函数名  ： Creat_LED_Task
  * @ 功能说明： 创建LED_Task任务
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static UINT32 Creat_LED_Task()
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 定义一个用于创建任务的参数结构体 */
    TSK_INIT_PARAM_S task_init_param;

    task_init_param.usTaskPrio = 5; /* 任务优先级，数值越小，优先级越高 */
    task_init_param.pcName = "LED_Task";/* 任务名 */
    task_init_param.pfnTaskEntry = (TSK_ENTRY_FUNC)LED_Task;/* 任务函数入口 */
    task_init_param.uwStackSize = 1024;     /* 堆栈大小 */

    uwRet = LOS_TaskCreate(&g_LED_Task_Handle, &task_init_param);/* 创建任务 */
    return uwRet;
}
/*******************************************************************
  * @ 函数名  ： Creat_Key_Task
  * @ 功能说明： 创建Key_Task任务
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static UINT32 Creat_Key_Task()
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;
    TSK_INIT_PARAM_S task_init_param;

    task_init_param.usTaskPrio = 4; /* 任务优先级，数值越小，优先级越高 */
    task_init_param.pcName = "Key_Task";    /* 任务名*/
    task_init_param.pfnTaskEntry = (TSK_ENTRY_FUNC)Key_Task;/* 任务函数入口 */
    task_init_param.uwStackSize = 1024; /* 堆栈大小 */

    uwRet = LOS_TaskCreate(&g_Key_Task_Handle, &task_init_param);/* 创建任务 */

    return uwRet;
}

/******************************************************************
  * @ 函数名  ： LED_Task
  * @ 功能说明： LED_Task任务实现
  * @ 参数    ： NULL
  * @ 返回值  ： NULL
  *****************************************************************/
static void LED_Task(void)
{
    /* 定义一个事件接收变量 */
    UINT32 uwRet;

    /* 任务都是一个无限循环，不能返回 */
    while (1)
    {
        /* 等待事件标志组 */
        uwRet = LOS_EventRead(&s_EventGroup_CB,          /* 事件标志组对象 */
                              KEY1_EVENT | KEY2_EVENT, /* 等待任务感兴趣的事件 */
                              LOS_WAITMODE_AND,       /* 等待两位均被置位 */
                              LOS_WAIT_FOREVER);          /* 无期限等待 */

        if ((uwRet & (KEY1_EVENT | KEY2_EVENT)) == (KEY1_EVENT | KEY2_EVENT))
        {
            /* 如果接收完成并且正确 */
            printf("KEY1 and KEY2 down\r\n");
            DEBUG_LED1_TOGGLE;              /* LED1 反转 */
            DEBUG_LED2_TOGGLE;              /* LED2 反转 */
            LOS_EventClear(&s_EventGroup_CB, ~KEY1_EVENT);  /* 清除事件标志 */
            LOS_EventClear(&s_EventGroup_CB, ~KEY2_EVENT);  /* 清除事件标志 */
        }
    }
}

/******************************************************************
  * @ 函数名  ： Key_Task
  * @ 功能说明： Key_Task任务实现
  * @ 参数    ： NULL
  * @ 返回值  ： NULL
  *****************************************************************/
static void Key_Task(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 任务都是一个无限循环，不能返回 */
    while (1)
    {
        /* KEY1 被按下 */
        if (Key_Scan(DEBUG_KEY1_GPIO_PORT, DEBUG_KEY1_GPIO_PIN))
        {
            printf("KEY1 down\r\n");
            /* 触发一个事件1 */
            LOS_EventWrite(&s_EventGroup_CB, KEY1_EVENT);
        }

        /* KEY2 被按下 */
        if (Key_Scan(DEBUG_KEY2_GPIO_PORT, DEBUG_KEY2_GPIO_PIN))
        {
            printf("KEY2 down\r\n");
            /* 触发一个事件2 */
            LOS_EventWrite(&s_EventGroup_CB, KEY2_EVENT);
        }

        LOS_TaskDelay(20);     /* 每20ms扫描一次 */
    }
}

/*******************************************************************
  * @ 函数名  ： BSP_Init
  * @ 功能说明： 板级外设初始化，所有板子上的初始化均可放在这个函数里面
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static void BSP_Init(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    debug_gpio_init();
    debug_uart_init();
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
