/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC_SingleConversion C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2021/09/03      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"
/* LiteOS 头文件 */
#include "los_sys.h"
#include "los_task.ph"
#include "los_sem.h"
/* 板级外设头文件 */
#include "debug_uart.h"
#include "debug_gpio.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */

static UINT32 AppTaskCreate(void);
static UINT32 Creat_Read_Task(void);
static UINT32 Creat_Write_Task(void);

static void Read_Task(void);
static void Write_Task(void);
static void BSP_Init(void);

/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */

/* 定义任务句柄 */
UINT32 g_Read_Task_Handle;
UINT32 g_Write_Task_Handle;

/* 定义二值信号量的句柄 */
UINT32 g_BinarySem_Handle;

uint8_t g_ucValue[2] = { 0x00U, 0x00U };

/* Private Function ---------------------------------------------------------- */

/***************************************************************
  * @brief  主函数
  * @param  无
  * @retval 无
  * @note   第一步：开发板硬件初始化
            第二步：创建APP应用任务
            第三步：启动LiteOS，开始多任务调度，启动失败则输出错误信息
  **************************************************************/
int main(void)
{
    UINT32 uwRet = LOS_OK;  /* 定义一个任务创建的返回值，默认为创建成功 */

    /* 板载相关初始化 */
    BSP_Init();

    printf("example start\r\n");

    /* LiteOS 内核初始化 */
    uwRet = LOS_KernelInit();

    if (uwRet != LOS_OK)
    {
        printf("[err]LiteOS core init err! code = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    uwRet = AppTaskCreate();

    if (uwRet != LOS_OK)
    {
        printf("[err]AppTaskCreate err! cde = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    /* 开启LiteOS任务调度 */
    LOS_Start();

    /* 正常情况下不会执行到这里 */
    while (1);

}


/*********************************************************************
  * @ 函数名  ： AppTaskCreate
  * @ 功能说明： 任务创建，为了方便管理，所有的任务创建函数都可以放在这个函数里面
  * @ 参数    ： 无
  * @ 返回值  ： 无
  *******************************************************************/
static UINT32 AppTaskCreate(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 创建一个二值信号量*/
    uwRet = LOS_BinarySemCreate(1, &g_BinarySem_Handle);

    if (uwRet != LOS_OK)
    {
        printf("BinarySem create err ! code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    uwRet = Creat_Read_Task();

    if (uwRet != LOS_OK)
    {
        printf("Read_Task create err! code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    uwRet = Creat_Write_Task();

    if (uwRet != LOS_OK)
    {
        printf("Write_Task create err! code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    return LOS_OK;
}


/******************************************************************
  * @ 函数名  ： Creat_Read_Task
  * @ 功能说明： 创建Read_Task任务
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static UINT32 Creat_Read_Task()
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 定义一个用于创建任务的参数结构体 */
    TSK_INIT_PARAM_S task_init_param;

    task_init_param.usTaskPrio = 5; /* 任务优先级，数值越小，优先级越高 */
    task_init_param.pcName = "Read_Task";/* 任务名 */
    task_init_param.pfnTaskEntry = (TSK_ENTRY_FUNC)Read_Task;/* 任务函数入口 */
    task_init_param.uwStackSize = 1024;     /* 堆栈大小 */

    uwRet = LOS_TaskCreate(&g_Read_Task_Handle, &task_init_param);/* 创建任务 */
    return uwRet;
}
/*******************************************************************
  * @ 函数名  ： Creat_Write_Task
  * @ 功能说明： 创建Write_Task任务
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static UINT32 Creat_Write_Task()
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;
    TSK_INIT_PARAM_S task_init_param;

    task_init_param.usTaskPrio = 4; /* 任务优先级，数值越小，优先级越高 */
    task_init_param.pcName = "Write_Task";  /* 任务名*/
    task_init_param.pfnTaskEntry = (TSK_ENTRY_FUNC)Write_Task;/* 任务函数入口 */
    task_init_param.uwStackSize = 1024; /* 堆栈大小 */

    uwRet = LOS_TaskCreate(&g_Write_Task_Handle, &task_init_param);/* 创建任务 */

    return uwRet;
}

/******************************************************************
  * @ 函数名  ： Read_Task
  * @ 功能说明： Read_Task任务实现
  * @ 参数    ： NULL
  * @ 返回值  ： NULL
  *****************************************************************/
static void Read_Task(void)
{
    /* 任务都是一个无限循环，不能返回 */
    while (1)
    {
        LOS_SemPend(g_BinarySem_Handle, LOS_WAIT_FOREVER);    /* 获取二值信号量 xSemaphore,没获取到则一直等待 */

        if (g_ucValue [ 0 ] == g_ucValue [ 1 ])
        {
            printf("\r\nSuccessful\r\n");
        }
        else
        {
            printf("\r\nFail\r\n");
        }

        LOS_SemPost(g_BinarySem_Handle);     /* 给出二值信号量 xSemaphore */

    }
}
/******************************************************************
  * @ 函数名  ： Write_Task
  * @ 功能说明： Write_Task任务实现
  * @ 参数    ： NULL
  * @ 返回值  ： NULL
  *****************************************************************/
static void Write_Task(void)
{
    /* 定义一个创建任务的返回类型，初始化为创建成功的返回值 */
    UINT32 uwRet = LOS_OK;

    /* 任务都是一个无限循环，不能返回 */
    while (1)
    {
        LOS_SemPend(g_BinarySem_Handle, LOS_WAIT_FOREVER);     /* 获取二值信号量 g_BinarySem_Handle，没获取到则一直等待 */
        g_ucValue [ 0 ] ++;
        DEBUG_LED1_TOGGLE
        LOS_TaskDelay(1000);                     /* 延时1000ms */
        g_ucValue [ 1 ] ++;
        DEBUG_LED2_TOGGLE
        LOS_SemPost(g_BinarySem_Handle);      /* 给出二值信号量 xSemaphore */
        LOS_TaskYield();                                    /* 放弃剩余时间片，进行一次任务切换 */
    }
}

/*******************************************************************
  * @ 函数名  ： BSP_Init
  * @ 功能说明： 板级外设初始化，所有板子上的初始化均可放在这个函数里面
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static void BSP_Init(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    debug_gpio_init();
    debug_uart_init();

}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
