/**********************************************************************************
 *
 * @file    .c
 * @brief
 *
 * @date
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2022.11.30      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
static md_spi_init_t s_es_lcd_spi;
static md_dma_descriptor_t s_dma0_ctrl_base[1] __attribute__((aligned(512)));

/* Public Variables ---------------------------------------------------------- */

md_dma_config_t g_lcd_spi_dma_tx_config;
volatile uint8_t g_es_lvgl_lcd_tx_complete = 0U;
uint32_t g_es_lvgl_lcd_tx_num = 0U;
uint16_t *g_es_lvgl_lcd_tx_buf = NULL;
#if (ES_LVGL_USE_2_DISP_BUF)
    lv_disp_drv_t *g_es_dma_flush_disp_drv = NULL;
#endif /*(ES_LVGL_USE_2_DISP_BUF)*/

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

void es_init_mcu_pin(void)
{
    md_gpio_init_t gpio_init;

    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP_DOWN;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_20;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_20;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;

    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;

    gpio_init.func  = ES_LCD_SPI_SOFT_NSS_PIN_MD_FUNC;
    md_gpio_init(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_LCD_RST_PIN_MD_FUNC;
    md_gpio_init(ES_LCD_RST_PIN_PORT, ES_LCD_RST_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_LCD_CD_PIN_MD_FUNC;
    md_gpio_init(ES_LCD_CD_PIN_PORT, ES_LCD_CD_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_LCD_SPI_SCK_PIN_MD_FUNC;
    md_gpio_init(ES_LCD_SPI_SCK_PIN_PORT, ES_LCD_SPI_SCK_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_LCD_SPI_MOSI_PIN_MD_FUNC;
    md_gpio_init(ES_LCD_SPI_MOSI_PIN_PORT, ES_LCD_SPI_MOSI_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_TOUCH_RST_PIN_MD_FUNC;
    md_gpio_init(ES_TOUCH_RST_PIN_PORT, ES_TOUCH_RST_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_TOUCH_INT_PIN_MD_FUNC;
    md_gpio_init(ES_TOUCH_INT_PIN_PORT, ES_TOUCH_INT_PIN_MD_PIN, &gpio_init);

    gpio_init.odos  = MD_GPIO_OPEN_DRAIN;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;

    gpio_init.func  = ES_TOUCH_I2C_SDA_PIN_MD_FUNC;
    md_gpio_init(ES_TOUCH_I2C_SDA_PIN_PORT, ES_TOUCH_I2C_SDA_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_TOUCH_I2C_SCL_PIN_MD_FUNC;
    md_gpio_init(ES_TOUCH_I2C_SCL_PIN_PORT, ES_TOUCH_I2C_SCL_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_LCD_BL_PIN_MD_FUNC;
    md_gpio_init(ES_LCD_BL_PIN_PORT, ES_LCD_BL_PIN_MD_PIN, &gpio_init);        
    md_gpio_set_pin_high(ES_LCD_BL_PIN_PORT, ES_LCD_BL_PIN_MD_PIN);
}

void es_init_touch_exti(void)
{
    md_gpio_set_interrupt_port(ES_TOUCH_INT_PIN_PORT, ES_TOUCH_INT_PIN_MD_PIN);      /* set exit interrupt port */
    md_gpio_enable_trailing_edge_trigger(ES_TOUCH_INT_PIN_MD_PIN); /* enable riging edge trigger */
    md_gpio_enable_external_interrupt(ES_TOUCH_INT_PIN_MD_PIN);  /* enable exti */
    md_gpio_clear_flag(ES_TOUCH_INT_PIN_MD_PIN);
    md_mcu_irq_config(ES_TOUCH_INT_IRQn, 3, 3, ENABLE);       /* enable NVIC EXTI_IRQn */
}

#if ((ES_TOUCH_USE_SOFT_I2C) == 0)
void es_init_touch_i2c(void)
{
    md_i2c_init_t es_touch_i2c;

    es_touch_i2c.module       = MD_I2C_MODULE_MASTER;
    es_touch_i2c.clk_speed    = 400000;
    es_touch_i2c.own_addr1    = 0xBA << 1;
    es_touch_i2c.addr_mode    = MD_I2C_ADDR_7BIT;
    es_touch_i2c.general_call = MD_I2C_GENERALCALL_DISABLE;
    es_touch_i2c.no_stretch   = MD_I2C_NOSTRETCH_DISABLE;

    md_i2c_init(ES_TOUCH_I2C_PERIPHERAL, &es_touch_i2c);  /*According to the initialization parameter I2C module*/
}

uint8_t es_touch_gt911_write_data(uint16_t addr, uint8_t value)
{
    uint16_t cnt;

    md_i2c_set_transmit_fifo_reset(ES_TOUCH_I2C_PERIPHERAL);   /*Reset send FIFO*/

    md_i2c_enable_master_write(ES_TOUCH_I2C_PERIPHERAL);   /*Host write data model*/
    md_i2c_set_transmit_length(ES_TOUCH_I2C_PERIPHERAL, 3); /*The transmission of bytes*/
    md_i2c_set_slave_addr(ES_TOUCH_I2C_PERIPHERAL, ES_GT911_I2C_WADDR);  /*Set from the machine address*/
    md_i2c_set_start(ES_TOUCH_I2C_PERIPHERAL);     /*+ 7 bus spare time to send a start bit bit from machine address + 1 write a bit*/

    md_i2c_set_tx_reg_data(ES_TOUCH_I2C_PERIPHERAL, (uint8_t)(addr >> 8));

    cnt = 10000;

    while (md_i2c_is_active_txf(ES_TOUCH_I2C_PERIPHERAL) && (--cnt));

    md_i2c_set_tx_reg_data(ES_TOUCH_I2C_PERIPHERAL, (uint8_t)(addr & 0xFF));

    cnt = 10000;

    while (md_i2c_is_active_txf(ES_TOUCH_I2C_PERIPHERAL) && (--cnt));

    md_i2c_set_tx_reg_data(ES_TOUCH_I2C_PERIPHERAL, value);

    cnt = 10000;

    while (!md_i2c_is_active_tc(ES_TOUCH_I2C_PERIPHERAL) && (--cnt));  /*The transfer to complete*/

    md_i2c_set_stop(ES_TOUCH_I2C_PERIPHERAL);  /*Send the stop bit*/

    if (cnt)
        return 0;
    else
        return 1;
}

uint8_t es_touch_gt911_read_data(uint16_t addr, uint8_t rx_cnt, uint8_t *rx_buf)
{
    uint8_t i;
    uint16_t cnt;

    md_i2c_set_transmit_fifo_reset(ES_TOUCH_I2C_PERIPHERAL);   /*Reset send FIFO*/
    md_i2c_set_receive_fifo_reset(ES_TOUCH_I2C_PERIPHERAL);    /*Reset receive FIFO*/

    md_i2c_enable_master_write(ES_TOUCH_I2C_PERIPHERAL);   /*Host write data model*/
    md_i2c_set_transmit_length(ES_TOUCH_I2C_PERIPHERAL, 2); /*The transmission of bytes*/
    md_i2c_set_slave_addr(ES_TOUCH_I2C_PERIPHERAL, ES_GT911_I2C_WADDR);  /*Set from the machine address*/
    md_i2c_set_start(ES_TOUCH_I2C_PERIPHERAL);     /*+ 7 bus spare time to send a start bit bit from machine address + 1 write a bit*/

    md_i2c_set_tx_reg_data(ES_TOUCH_I2C_PERIPHERAL, (uint8_t)(addr >> 8));

    cnt = 10000;

    while (md_i2c_is_active_txf(ES_TOUCH_I2C_PERIPHERAL) && (--cnt));

    md_i2c_set_tx_reg_data(ES_TOUCH_I2C_PERIPHERAL, (uint8_t)(addr & 0xFF));

    cnt = 10000;

    while (!md_i2c_is_active_tc(ES_TOUCH_I2C_PERIPHERAL) && (--cnt));  /*Waiting for completion of the inside address*/

    md_i2c_enable_master_read(ES_TOUCH_I2C_PERIPHERAL);    /*Host read data model*/
    md_i2c_set_slave_addr(ES_TOUCH_I2C_PERIPHERAL, ES_GT911_I2C_RADDR);  /*Set from the machine address*/
    md_i2c_set_transmit_length(ES_TOUCH_I2C_PERIPHERAL, rx_cnt); /*The transmission of bytes*/
    md_i2c_set_start(ES_TOUCH_I2C_PERIPHERAL);     /*restart*/

    for (i = 0; i < rx_cnt; i++)
    {
        cnt = 10000;

        while (!md_i2c_is_active_rxth(ES_TOUCH_I2C_PERIPHERAL) && (--cnt));  /*Waiting to receive complete*/

        rx_buf[i] = md_i2c_get_rx_reg_data(ES_TOUCH_I2C_PERIPHERAL);
    }

    cnt = 10000;

    while (!md_i2c_is_active_tc(ES_TOUCH_I2C_PERIPHERAL) && (--cnt));  /*The transfer to complete*/

    md_i2c_set_stop(ES_TOUCH_I2C_PERIPHERAL);  /*Send the stop bit*/

    if (cnt)
        return 0;
    else
        return 1;
}
#endif /*((ES_TOUCH_USE_SOFT_I2C) == 0)*/

void es_init_lcd_spi_dma(void)
{
    NVIC_SetPriority(DMA_IRQn, 2);
    NVIC_EnableIRQ(DMA_IRQn);
    memset(s_dma0_ctrl_base, 0x0, sizeof(s_dma0_ctrl_base));

    memset(&g_lcd_spi_dma_tx_config, 0x0, sizeof(md_dma_config_t));
    md_dma_set_ctrlbase((uint32_t)s_dma0_ctrl_base);

    /* Enable DMA */
    md_dma_enable();

    md_dma_config_struct(&g_lcd_spi_dma_tx_config);
    g_lcd_spi_dma_tx_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;
    g_lcd_spi_dma_tx_config.src_inc    = MD_DMA_DATA_INC_HALFWORD;
    g_lcd_spi_dma_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;
    g_lcd_spi_dma_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_lcd_spi_dma_tx_config.primary    = ENABLE;
    g_lcd_spi_dma_tx_config.burst      = ENABLE;
    g_lcd_spi_dma_tx_config.high_prio  = DISABLE;
    g_lcd_spi_dma_tx_config.interrupt  = ENABLE;
    g_lcd_spi_dma_tx_config.channel    = 0;
    g_lcd_spi_dma_tx_config.src        = (void *)(0);
    g_lcd_spi_dma_tx_config.dst        = (void *)(&ES_LCD_SPI_PERIPHERAL->DATA);
    g_lcd_spi_dma_tx_config.size       = 1024;
    g_lcd_spi_dma_tx_config.msel       = MD_DMA_MSEL_SPI1;
    g_lcd_spi_dma_tx_config.msigsel    = MD_DMA_MSIGSEL_SPI_TXEMPTY;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_lcd_spi_dma_tx_config);

    md_spi_enable_txdma(ES_LCD_SPI_PERIPHERAL);
}

void es_init_lcd_spi(void)
{
    s_es_lcd_spi.SPIx      = ES_LCD_SPI_PERIPHERAL;                 /*Using SPI0*/
    s_es_lcd_spi.mode      = MD_SPI_MODE_MASTER;   /*SPI host mode*/
    s_es_lcd_spi.baud      = MD_SPI_BAUD_4;        /*clock / 8*/
    s_es_lcd_spi.data_size = MD_SPI_DATA_SIZE_8;   /*8 bit pattern*/
    s_es_lcd_spi.polarity  = MD_SPI_CPOL_HIGH;     /*Free high level*/
    s_es_lcd_spi.phase     = MD_SPI_CPHA_SECOND;   /*The second edge receiving data*/
    s_es_lcd_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /*Send the MSB first*/
    s_es_lcd_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_es_lcd_spi.ss_en     = DISABLE;
    s_es_lcd_spi.crc_calc  = DISABLE;
    s_es_lcd_spi.crc_poly  = 0;

    md_spi_init(&s_es_lcd_spi);                    /*According to the parameter initialization SPI peripherals*/
}

void es_st7789_port_lcd_wrte_cmd(uint8_t cmd)
{
    ES_LCD1_SET_CMD()
    ES_LCD_SPI_CS_OUTPUT_LOW()
    md_spi_send_byte_fast(&s_es_lcd_spi, cmd);
    ES_LCD_SPI_CS_OUTPUT_HIGH()
}

void es_st7789_port_lcd_wrte_data(uint8_t data)
{
    ES_LCD1_SET_DATA()
    ES_LCD_SPI_CS_OUTPUT_LOW()
    md_spi_send_byte_fast(&s_es_lcd_spi, data);
    ES_LCD_SPI_CS_OUTPUT_HIGH()
}

#if (ES_LVGL_USE_2_DISP_BUF)
void es_lv_st7789_write_area(lv_disp_drv_t *disp_drv, const lv_area_t *area, lv_color_t *color_p)
{
    uint32_t size = (area->x2 - area->x1 + 1) * (area->y2 - area->y1 + 1);
    uint8_t temp8;
    uint32_t cnt = 10000000U;

    /*等待DMA完成*/
    while ((!g_es_lvgl_lcd_tx_complete) && (cnt--))
    {
    }

    /*等待SPI传输完成*/
    while (((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_TXE_MSK) == 0)) {}

    while (((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_BUSY_MSK) == SPI_STAT_BUSY_MSK)) {}

    /*读取SPI接收FIFO直到空*/
    while ((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_RXE_MSK) == RESET)
    {
        temp8 = (uint8_t)ES_LCD_SPI_PERIPHERAL->DATA;
    }

    ES_LCD_SPI_CS_OUTPUT_HIGH()

    /*SPI数据帧长度->8*/
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_FLEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);

    es_st7789_lcd_write_area_cmd(area->x1, area->x2, area->y1, area->y2);

    /*SPI数据帧长度->16*/
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_FLEN_MSK);
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);

    /*LCD写像素数据+SPI片选*/
    ES_LCD1_SET_DATA()
    ES_LCD_SPI_CS_OUTPUT_LOW()

    /*SPI发送DMA开始*/

    g_es_dma_flush_disp_drv = disp_drv;  /*LCD 显示*/
    g_es_lvgl_lcd_tx_num = size;
    g_es_lvgl_lcd_tx_buf = (uint16_t *)color_p;
    md_dma_enable_channel(MD_DMA_CH_0);

    (void)temp8;
}
#else
void es_lv_st7789_write_area(lv_disp_drv_t *disp_drv, const lv_area_t *area, lv_color_t *color_p)
{
    uint32_t size = (area->x2 - area->x1 + 1) * (area->y2 - area->y1 + 1);
    uint8_t temp8;
    uint32_t cnt = 10000000U;

    es_st7789_lcd_write_area_cmd(area->x1, area->x2, area->y1, area->y2);

    /*SPI数据帧长度->16*/
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_FLEN_MSK);
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);

    /*LCD写像素数据+SPI片选*/
    ES_LCD1_SET_DATA()
    ES_LCD_SPI_CS_OUTPUT_LOW()

    /*SPI发送DMA开始*/
    g_es_lvgl_lcd_tx_complete = 0;
    g_es_lvgl_lcd_tx_num = size;
    g_es_lvgl_lcd_tx_buf = (uint16_t *)color_p;
    md_dma_enable_channel(MD_DMA_CH_0);

    /*等待DMA完成*/
    while ((!g_es_lvgl_lcd_tx_complete) && (--cnt))
    {
    }

    /*等待SPI传输完成*/
    while (((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_TXE_MSK) == 0)) {}

    while (((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_BUSY_MSK) == SPI_STAT_BUSY_MSK)) {}

    /*读取SPI接收FIFO直到空*/
    while ((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_RXE_MSK) == RESET)
    {
        temp8 = (uint8_t)ES_LCD_SPI_PERIPHERAL->DATA;
    }

    ES_LCD_SPI_CS_OUTPUT_HIGH()

    /*SPI数据帧长度->8*/
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_FLEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);

    (void)temp8;

    if (disp_drv)
        lv_disp_flush_ready(disp_drv);  /*LCD 显示完成*/
}
#endif /*(ES_LVGL_USE_2_DISP_BUF)*/

void es_init_touch_gt91x(void)
{
#if ((ES_TOUCH_USE_SOFT_I2C) == 0)
    es_init_touch_i2c();
#endif /*((ES_TOUCH_USE_SOFT_I2C) == 0)*/
    es_init_touch_exti();
    gt911_touch_reset();
    gt911_touch_default_config();
}

void es_init_lcd_st7789(void)
{
    es_init_lcd_spi();
    es_init_lcd_spi_dma();
    es_st7789_lcd_init();
}

void es_mcu_init(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);   /*Can make the frequency doubling, by the crystal three times the frequency of vibration to 48 MHZ*/
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);       /*Choose the frequency doubling the clock for the system clock*/

    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();       /*Can make all the peripheral clock*/
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    es_init_mcu_pin();

    es_init_lcd_st7789();
    es_init_touch_gt91x();
}
