/**********************************************************************************
 *
 * @file    .c
 * @brief
 *
 * @date
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2022.11.30      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "es_gt91x.h"

/* Private Macros ------------------------------------------------------------ */
#define GT911_READ_ADDR 0x814E
#define GT911_ID_ADDR   0x8140

/* Private Variables --------------------------------------------------------- */
/*gt911  reg_0x8047_0x8100*/
static const uint8_t s_es_gt911_config[] =
{
    0x41, (ES_TOUCH_MAX_X_VALUE & 0xFF), ((ES_TOUCH_MAX_X_VALUE >> 8) & 0xFF),
    (ES_TOUCH_MAX_Y_VALUE & 0xFF), ((ES_TOUCH_MAX_Y_VALUE >> 8) & 0xFF), (ES_MAX_TOUCH_POINT_NUM),
    ((ES_TOUCH_CTRL_X2Y << 3) | (0x4) | ES_TOUCH_INT_MODE), 0x00, \
    0x01, 0x08, 0x28, 0x05, (ES_TOUCH_SCREEN_POINT_CTRL_LEVEL1), (ES_TOUCH_SCREEN_POINT_CTRL_LEVEL2),
    0x0F, (ES_TOUCH_INT_PERIOD), \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x87, 0x28, 0x0a, 0x17, 0x15, \
    0x31, 0x0d, 0x00, 0x00, 0x00, 0x9a, 0x04, 0x25, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x64, 0x32, \
    0x00, 0x00, 0x00, 0x0f, 0x94, 0x94, 0xc5, 0x02, \
    0x07, 0x00, 0x00, 0x04, 0x8d, 0x13, 0x00, 0x5c, \
    0x1e, 0x00, 0x3b, 0x30, 0x00, 0x28, 0x4c, 0x00, \
    0x1c, 0x78, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x14, 0x12, 0x10, 0x0e, 0x0c, 0x0a, 0x08, 0x06, \
    0x04, 0x02, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, \
    0x04, 0x06, 0x08, 0x0a, 0x0c, 0x26, 0x24, 0x22, \
    0x21, 0x20, 0x1f, 0x1e, 0x1d, 0xff, 0xff, 0xff, \
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x3c, 0x00
};

static const uint16_t s_es_gt911_tp_reg[] = {0x814F, 0x8157, 0x815F, 0x8167, 0x816F, 0x8177, 0x817F, 0x8187, 0x818F, 0x8197};

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  GT911 scan screen
  * @retval 0, no touch 1, touched
  */
uint8_t gt911_touch_scan(void *a_buf, uint32_t read_num)
{
    uint8_t buf[7], i, temp = 0U;
    struct es_touch_data *es32_read_data = (struct es_touch_data *)a_buf;
    uint32_t timestamp = md_get_tick();

    es_touch_gt911_read_data(GT911_READ_ADDR, 1, buf);

    if (!(buf[0] & 0x80))
        return 0;

    temp = buf[0] & 0XF;

    if ((temp) && (temp <= ES_MAX_TOUCH_POINT_NUM))
    {
        temp = (read_num > temp) ? (temp) : (read_num);

        for (i = 0; i < temp; i++)
        {
            es_touch_gt911_read_data(s_es_gt911_tp_reg[i], 7, buf);

            es32_read_data[i].timestamp    = timestamp;
            es32_read_data[i].track_id     = buf[0];

#if ES_LCD_X2Y
#if ES_LCD_X_ORDER
#if ES_LCD_Y_ORDER
            es32_read_data[i].y_coordinate = ES_LCD_WIDTH - (((uint16_t)buf[2] << 8) + buf[1]);
            es32_read_data[i].x_coordinate = ES_LCD_LENGTH - (((uint16_t)buf[4] << 8) + buf[3]);
#else
            es32_read_data[i].y_coordinate = ES_LCD_WIDTH - (((uint16_t)buf[2] << 8) + buf[1]);
            es32_read_data[i].x_coordinate = ((uint16_t)buf[4] << 8) + buf[3];
#endif /*ES_LCD_Y_ORDER*/
#else
#if ES_LCD_Y_ORDER
            es32_read_data[i].y_coordinate = ((uint16_t)buf[2] << 8) + buf[1];
            es32_read_data[i].x_coordinate = ES_LCD_LENGTH - (((uint16_t)buf[4] << 8) + buf[3]);
#else
            es32_read_data[i].y_coordinate = ((uint16_t)buf[2] << 8) + buf[1];
            es32_read_data[i].x_coordinate = ((uint16_t)buf[4] << 8) + buf[3];
#endif /*ES_LCD_Y_ORDER*/
#endif /*ES_LCD_X_ORDER*/
#else
#if ES_LCD_X_ORDER
#if ES_LCD_Y_ORDER
            es32_read_data[i].x_coordinate = ES_LCD_WIDTH - (((uint16_t)buf[2] << 8) + buf[1]);
            es32_read_data[i].y_coordinate = ES_LCD_LENGTH - (((uint16_t)buf[4] << 8) + buf[3]);
#else
            es32_read_data[i].x_coordinate = ES_LCD_WIDTH - (((uint16_t)buf[2] << 8) + buf[1]);
            es32_read_data[i].y_coordinate = ((uint16_t)buf[4] << 8) + buf[3];
#endif /*ES_LCD_Y_ORDER*/
#else
#if ES_LCD_Y_ORDER
            es32_read_data[i].x_coordinate = ((uint16_t)buf[2] << 8) + buf[1];
            es32_read_data[i].y_coordinate = ES_LCD_LENGTH - (((uint16_t)buf[4] << 8) + buf[3]);
#else
            es32_read_data[i].x_coordinate = ((uint16_t)buf[2] << 8) + buf[1];
            es32_read_data[i].y_coordinate = ((uint16_t)buf[4] << 8) + buf[3];
#endif /*ES_LCD_Y_ORDER*/
#endif /*ES_LCD_X_ORDER*/
#endif /*ES_LCD_X2Y*/
            es32_read_data[i].width        = ((uint16_t)buf[6] << 8) + buf[5];
        }
    }

    es_touch_gt911_write_data(GT911_READ_ADDR, 0U);

    return temp;
}

void gt911_touch_reset(void)
{
    /*INT 管脚变为输出状态*/
    ES_TOUCH_INT_PIN_PORT->MODE &= (~(0x3 * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN));
    ES_TOUCH_INT_PIN_PORT->MODE |= (MD_GPIO_MODE_OUTPUT * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN);

    ES_TOUCH_INT_PIN_PORT->PUPD &= (~(0x3 * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN));
    ES_TOUCH_INT_PIN_PORT->PUPD |= (MD_GPIO_PUSH_UP_DOWN * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN);

    md_gpio_set_pin_high(ES_TOUCH_RST_PIN_PORT, ES_TOUCH_RST_PIN_MD_PIN);
    md_gpio_set_pin_high(ES_TOUCH_INT_PIN_PORT, ES_TOUCH_INT_PIN_MD_PIN);

    ES_DELAY_MS(100);

    md_gpio_set_pin_low(ES_TOUCH_RST_PIN_PORT, ES_TOUCH_RST_PIN_MD_PIN);
    md_gpio_set_pin_low(ES_TOUCH_INT_PIN_PORT, ES_TOUCH_INT_PIN_MD_PIN);

    ES_DELAY_MS(100);

    md_gpio_set_pin_high(ES_TOUCH_RST_PIN_PORT, ES_TOUCH_RST_PIN_MD_PIN);

    ES_DELAY_MS(10);

    /*INT 管脚变为输入状态*/
    ES_TOUCH_INT_PIN_PORT->MODE &= (~(0x3 * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN));
    ES_TOUCH_INT_PIN_PORT->MODE |= (MD_GPIO_MODE_INPUT * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN);

    ES_TOUCH_INT_PIN_PORT->PUPD &= (~(0x3 * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN));
    ES_TOUCH_INT_PIN_PORT->PUPD |= (MD_GPIO_FLOATING * ES_TOUCH_INT_PIN_MD_PIN * ES_TOUCH_INT_PIN_MD_PIN);
}

void gt911_touch_default_config(void)
{
    uint32_t i;
    uint8_t sum;

    ES_DELAY_MS(50);   /*当主控将自身 INT 转化为悬浮输入态后，需要延时 50ms 再发送配置信息。*/

    es_touch_gt911_write_data(0x8040, 0x02);    /*软件复位*/

    es_touch_gt911_read_data(0x8047, 1, &sum);  /*读版本号*/

    for (i = 1; i < 184; i++)
    {
        es_touch_gt911_write_data(0x8047 + i, s_es_gt911_config[i]);
        sum += s_es_gt911_config[i];
    }

    es_touch_gt911_write_data(0x80FF, (uint8_t)((~sum) + 1)); /*配置信息校验*/
    es_touch_gt911_write_data(0x8100, 0x01);      /*配置更新标志*/

    ES_DELAY_MS(10);

    es_touch_gt911_write_data(0x8040, 0x00);    /*读坐标状态*/

    ES_DELAY_MS(100);
}

#if (ES_TOUCH_USE_SOFT_I2C)

#warning "using soft i2c!"
#define ES_SOFT_I2C_SCL_1   ES_TOUCH_I2C_SCL_PIN_PORT->BSRR = (ES_TOUCH_I2C_SCL_PIN_MD_PIN << GPIO_BSRR_BSR_POSS)
#define ES_SOFT_I2C_SCL_0   ES_TOUCH_I2C_SCL_PIN_PORT->BSRR = (ES_TOUCH_I2C_SCL_PIN_MD_PIN << GPIO_BSRR_BRR_POSS)
#define ES_SOFT_I2C_SDA_1   ES_TOUCH_I2C_SDA_PIN_PORT->BSRR = (ES_TOUCH_I2C_SDA_PIN_MD_PIN << GPIO_BSRR_BSR_POSS)
#define ES_SOFT_I2C_SDA_0   ES_TOUCH_I2C_SDA_PIN_PORT->BSRR = (ES_TOUCH_I2C_SDA_PIN_MD_PIN << GPIO_BSRR_BRR_POSS)
#define TOUCH_READ_SDA      ((ES_TOUCH_I2C_SDA_PIN_PORT->DIN)&ES_TOUCH_I2C_SDA_PIN_MD_PIN)

void touch_sda_out(void)
{
    ES_TOUCH_I2C_SDA_PIN_PORT->MODE &= (~(0x3 * ES_TOUCH_I2C_SDA_PIN_MD_PIN * ES_TOUCH_I2C_SDA_PIN_MD_PIN));
    ES_TOUCH_I2C_SDA_PIN_PORT->MODE |= (MD_GPIO_MODE_OUTPUT * ES_TOUCH_I2C_SDA_PIN_MD_PIN * ES_TOUCH_I2C_SDA_PIN_MD_PIN);
}

void touch_sda_in(void)
{
    ES_TOUCH_I2C_SDA_PIN_PORT->MODE &= (~(0x3 * ES_TOUCH_I2C_SDA_PIN_MD_PIN * ES_TOUCH_I2C_SDA_PIN_MD_PIN));
    ES_TOUCH_I2C_SDA_PIN_PORT->MODE |= (MD_GPIO_MODE_INPUT * ES_TOUCH_I2C_SDA_PIN_MD_PIN * ES_TOUCH_I2C_SDA_PIN_MD_PIN);
}

void es_soft_i2c_start(void)
{
    touch_sda_out();
    ES_SOFT_I2C_SDA_1;
    ES_SOFT_I2C_SCL_1;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SDA_0;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SCL_0;
}

void es_soft_i2c_stop(void)
{
    touch_sda_out();
    ES_SOFT_I2C_SCL_0;
    ES_SOFT_I2C_SDA_0;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SCL_1;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SDA_1;
}

uint8_t es_soft_i2c_wait_ack(void)
{
    uint8_t et = 0U;

    touch_sda_in();
    ES_SOFT_I2C_SDA_1;
    __NOP();
    __NOP();
    ES_SOFT_I2C_SCL_1;
    ES_DELAY_US(5);

    while (TOUCH_READ_SDA)
    {
        et++;

        if (et > 250)
        {
            es_soft_i2c_stop();
            return 1;
        }

        ES_DELAY_US(5);
    }

    ES_SOFT_I2C_SCL_0;

    return 0;
}

void es_soft_i2c_ack(void)
{
    ES_SOFT_I2C_SCL_0;
    touch_sda_out();
    ES_SOFT_I2C_SDA_0;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SCL_1;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SCL_0;
}

void es_soft_i2c_nack(void)
{
    ES_SOFT_I2C_SCL_0;
    touch_sda_out();
    ES_SOFT_I2C_SDA_1;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SCL_1;
    ES_DELAY_US(5);
    ES_SOFT_I2C_SCL_0;
}

void es_soft_i2c_send_byte(uint8_t dat)
{
    uint8_t i;

    touch_sda_out();
    ES_SOFT_I2C_SCL_0;

    for (i = 0; i < 8; i++)
    {
        if ((dat & 0x80) >> 7)
            ES_SOFT_I2C_SDA_1;
        else
            ES_SOFT_I2C_SDA_0;

        dat <<= 1;
        ES_DELAY_US(5);
        ES_SOFT_I2C_SCL_1;
        ES_DELAY_US(5);
        ES_SOFT_I2C_SCL_0;
        ES_DELAY_US(5);
    }
}

uint8_t es_soft_i2c_read_byte(uint8_t ack)
{
    uint8_t i, rec = 0U;

    touch_sda_in();
    ES_DELAY_US(5);

    for (i = 0; i < 8; i++)
    {
        ES_SOFT_I2C_SCL_0;
        ES_DELAY_US(5);
        ES_SOFT_I2C_SCL_1;
        rec <<= 1;

        if (TOUCH_READ_SDA) rec++;

        ES_DELAY_US(5);
    }

    if (!ack)
        es_soft_i2c_nack();
    else
        es_soft_i2c_ack();

    return rec;
}

void gt911_write_prepare(uint16_t addr)
{
    es_soft_i2c_start();
    es_soft_i2c_send_byte(ES_GT911_I2C_WADDR);
    es_soft_i2c_wait_ack();
    es_soft_i2c_send_byte((uint8_t)(addr >> 8));
    es_soft_i2c_wait_ack();
    es_soft_i2c_send_byte((uint8_t)addr);
    es_soft_i2c_wait_ack();
}

uint8_t es_touch_gt911_write_data(uint16_t addr, uint8_t value)
{
    es_soft_i2c_start();
    gt911_write_prepare(addr);
    es_soft_i2c_send_byte(value);
    es_soft_i2c_wait_ack();
    es_soft_i2c_stop();
    return 0;
}

uint8_t es_touch_gt911_read_data(uint16_t addr, uint8_t cnt, uint8_t *desbuf)
{
    uint8_t res = 0U;
    uint8_t i;

    es_soft_i2c_start();
    gt911_write_prepare(addr);
    es_soft_i2c_start();
    es_soft_i2c_send_byte(ES_GT911_I2C_RADDR);
    es_soft_i2c_wait_ack();

    for (i = 0; i < cnt; i++)
    {
        if (i == (cnt - 1))
        {
            desbuf[i] = es_soft_i2c_read_byte(0);
        }
        else
        {
            desbuf[i] = es_soft_i2c_read_byte(1);
        }
    }

    es_soft_i2c_stop();
    res = 1;

    return res;
}
#endif /*(ES_TOUCH_USE_SOFT_I2C)*/
