/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          21 Oct 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
static TaskHandle_t g_task_app_handle = NULL;
static TaskHandle_t g_task_high_pri_handle = NULL;
static TaskHandle_t g_task_medium_pri_handle = NULL;
static TaskHandle_t g_task_low_pri_handle = NULL;
static uint32_t g_high_pri_cycle = 0U;
static uint32_t g_medium_pri_cycle = 0U;
static uint32_t g_low_pri_cycle = 0U;
static SemaphoreHandle_t g_mutex_test;
/* Private function prototypes ------------------------------------------------*/
static void task_app(void *);
static void task_high_pri(void *pvParameters);
static void task_medium_pri(void *pvParameters);
static void task_low_pri(void *pvParameters);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  Initializate pin of lpuart module.
  * @retval None
  */
void lpuart_pin_init(void)
{
    md_gpio_init_t init;

    /* Initialize tx pin */
    init.mode  = MD_GPIO_MODE_OUTPUT;
    init.odos  = MD_GPIO_PUSH_PULL;
    init.pupd  = MD_GPIO_PUSH_UP;
    init.podrv = MD_GPIO_OUT_DRIVE_6;
    init.nodrv = MD_GPIO_OUT_DRIVE_6;
    init.flt   = MD_GPIO_FILTER_DISABLE;
    init.type  = MD_GPIO_TYPE_CMOS;
    init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(LPUART0_TX_PORT, LPUART0_TX_PIN, &init);

    /* Initialize rx pin */
    init.mode  = MD_GPIO_MODE_INPUT;
    init.odos  = MD_GPIO_PUSH_PULL;
    init.pupd  = MD_GPIO_PUSH_UP;
    init.podrv = MD_GPIO_OUT_DRIVE_6;
    init.nodrv = MD_GPIO_OUT_DRIVE_6;
    init.flt   = MD_GPIO_FILTER_DISABLE;
    init.type  = MD_GPIO_TYPE_CMOS;
    init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(LPUART0_RX_PORT, LPUART0_RX_PIN, &init);

    return;
}

/**
  * @brief  Initializate lpuart module.
  * @retval None
  */
void lpuart_init()
{
    md_lpuart_init_t init;
    /* clear md_lpuart_init_t structure */
    memset(&init, 0x0, sizeof(md_lpuart_init_t));
    /* Initialize LPUART */
    init.baud        = 9600;
    init.word_length = MD_LPUART_WORD_LENGTH_8B;
    init.stop_bits   = MD_LPUART_STOP_BITS_1;
    init.parity      = MD_LPUART_PARITY_NONE;
    init.mode        = MD_LPUART_MODE_UART;
    init.fctl        = MD_LPUART_FLOW_CTL_DISABLE;
    md_lpuart_init(LPUART0, &init);
}

/**
  * @brief  Initializate pin of leds.
  * @retval None
  */
void led_pin_init()
{
    md_gpio_init_t init;

    init.mode  = MD_GPIO_MODE_OUTPUT;
    init.odos  = MD_GPIO_PUSH_PULL;
    init.pupd  = MD_GPIO_PUSH_UP;
    init.podrv = MD_GPIO_OUT_DRIVE_6;
    init.nodrv = MD_GPIO_OUT_DRIVE_6;
    init.flt   = MD_GPIO_FILTER_DISABLE;
    init.type  = MD_GPIO_TYPE_CMOS;
    init.func  = MD_GPIO_FUNC_1;
    
    md_gpio_init(LED1_PORT, LED1_PIN, &init);
    md_gpio_init(LED2_PORT, LED2_PIN, &init);
    md_gpio_set_pin_high(LED1_PORT,LED1_PIN);
    md_gpio_set_pin_high(LED2_PORT,LED2_PIN);
}
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    BaseType_t retval;
    
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC_6, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    
    /* Enable all peripherals clock */
    md_cmu_perh_clock_config(MD_CMU_PERH_ALL, ENABLE);
    
    /* Set LP16T0 clock source(LOSC) */
    SYSCFG_UNLOCK();
    md_cmu_set_lpuart0_clock_source(0x6);
    SYSCFG_LOCK();
    __enable_irq();
    
    /* Initialize LPUART */ 
    lpuart_pin_init();
    lpuart_init();
    /* Initialize LED */
    led_pin_init();
    
    printf("ES32H040x FreeRTOS demo\r\nSysCoreClock=%d\r\n",md_system_clock);
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
    }
}

int fputc(int ch,FILE*fp)
{
    UNUSED(fp);
    while (!md_lpuart_get_state_txemp(LPUART0));
    md_lpuart_send_data(LPUART0,(uint8_t)ch); 
    while (md_lpuart_get_state_txemp(LPUART0));
    return 0;
}


/**
  * @brief  App task creator
  * @param  parameter: Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();
    /* Create mutex or binary semaphore */
#if 1
    g_mutex_test = xSemaphoreCreateMutex();
#else
    g_mutex_test = xSemaphoreCreateBinary();
    xSemaphoreGive(g_mutex_test);
#endif /* Set to 1 to use mutex, 0 to use binary semaphore */

    if (g_mutex_test != NULL)
        printf("Create mutex OK\r\n");

    /* Create high priority task */
    retval = xTaskCreate(task_high_pri, "HighPriTask", 128, NULL, 5, &g_task_high_pri_handle);

    if (pdPASS == retval)
        printf("Task high priority Created\r\n");

    /* Create medium priority task*/
    retval = xTaskCreate(task_medium_pri, "MediumPriTask", 128, NULL, 4, &g_task_medium_pri_handle);

    if (pdPASS == retval)
        printf("Task medium priority Created\r\n");

    /* Create low priority task*/
    retval = xTaskCreate(task_low_pri, "LowPriTask", 128, NULL, 3, &g_task_low_pri_handle);

    if (pdPASS == retval)
        printf("Task low priority Created\r\n");

    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  High priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_high_pri(void *param)
{
    while (1)
    {
        /* Get mutex */
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) != pdTRUE)
        {
            printf("ERR:task_high_pri take failed\r\n");
        }

        printf("Mutex Take:  1\r\n");
        vTaskDelay(1000);

        if (xSemaphoreGive(g_mutex_test) != pdTRUE)
        {
            /* Toggle LED3 to indicate error */
            printf("ERR:task_high_pri give failed\r\n");
        }

        /* Keep count of the number of cycles this thread has performed */
        g_high_pri_cycle++;

        vTaskSuspend(NULL);
    }
}

/**
  * @brief  Medium priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_medium_pri(void *param)
{
    while (1)
    {
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) == pdTRUE)
        {

            printf("Mutex Take:    2\r\n");
            vTaskDelay(1000);

            if (eTaskGetState(g_task_high_pri_handle) != eSuspended)
            {
                /* Did not expect to execute until the high priority thread was
                suspended.
                Toggle LED3 to indicate error */
                printf("ERR:task_high_pri should be suspended!\r\n");
            }
            else
            {
                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(g_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_medium_pri give failed");
                }

                vTaskSuspend(NULL);
            }
        }
        else
        {
            /* We should not leave the osMutexWait() function
            until the mutex was obtained.
            Toggle LED3 to indicate error */
            printf("ERR:task_medium_pri cant take failed\r\n");
        }

        /* The High and Medium priority threads should be in lock step */
        if (g_high_pri_cycle != (g_medium_pri_cycle + 1))
        {
            printf("Lock step failed\r\n");
        }

        /* Print a message to indicate medium task is running */
        printf("Task2 Run :    2\r\n");

        /* Keep count of the number of cycles this task has performed so a
        stall can be detected */
        g_medium_pri_cycle++;
        vTaskDelay(500);
    }
}

/**
  * @brief  Low priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_low_pri(void *param)
{
    while (1)
    {
        /* Keep attempting to obtain the mutex.  We should only obtain it when
        the medium-priority thread has suspended itself, which in turn should only
        happen when the high-priority thread is also suspended */
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) == pdTRUE)
        {
            printf("Mutex Take:      3\r\n");
            vTaskDelay(1000);

            /* Is the haigh and medium-priority threads suspended? */
            if ((eTaskGetState(g_task_high_pri_handle) != eSuspended) || (eTaskGetState(g_task_medium_pri_handle) != eSuspended))
            {
                printf("ERR:task_high_pri and task_medium_pri should be suspended!\r\n");
            }
            else
            {
                /* Keep count of the number of cycles this task has performed
                so a stall can be detected */
                g_low_pri_cycle++;

                /* We can resume the other tasks here even though they have a
                higher priority than the this thread. When they execute they
                will attempt to obtain the mutex but fail because the low-priority
                thread is still the mutex holder.  this thread will then inherit
                the higher priority.  The medium-priority thread will block indefinitely
                when it attempts to obtain the mutex, the high-priority thread will only
                block for a fixed period and an error will be latched if the
                high-priority thread has not returned the mutex by the time this
                fixed period has expired */
                /* If binary semaphore is used, you will see "Task 2 run" before we give
                mutex to high priority task. Otherwise, Task 2 would not run before high
                priority task run*/
                vTaskResume(g_task_high_pri_handle);
                vTaskResume(g_task_medium_pri_handle);

                /* The other two tasks should now have executed and no longer
                be suspended */
                if ((eTaskGetState(g_task_high_pri_handle) == eSuspended) || (eTaskGetState(g_task_medium_pri_handle) == eSuspended))
                {
                    printf("ERR:task_high_pri and task_medium_pri should NOT be suspended!\r\n");
                }

                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(g_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_low_pri give failed");
                }
            }
        }
    }
}
/**
  * @}
  */
/**
  * @}
  */
