/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          21 Oct 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private define -------------------------------------------------------------*/
#define BIT_ALL 0xFFFFFFFF
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
static TaskHandle_t g_task_app_handle = NULL;
static TaskHandle_t g_task_led1_handle = NULL;
static TaskHandle_t g_task_led2_handle = NULL;
/* Private function prototypes ------------------------------------------------*/
static void task_app(void *);
static void task_led1(void *pvParameters);
static void task_led2(void *pvParameters);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  Initializate pin of lpuart module.
  * @retval None
  */
void lpuart_pin_init(void)
{
    md_gpio_init_t init;

    /* Initialize tx pin */
    init.mode  = MD_GPIO_MODE_OUTPUT;
    init.odos  = MD_GPIO_PUSH_PULL;
    init.pupd  = MD_GPIO_PUSH_UP;
    init.podrv = MD_GPIO_OUT_DRIVE_6;
    init.nodrv = MD_GPIO_OUT_DRIVE_6;
    init.flt   = MD_GPIO_FILTER_DISABLE;
    init.type  = MD_GPIO_TYPE_CMOS;
    init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(LPUART0_TX_PORT, LPUART0_TX_PIN, &init);

    /* Initialize rx pin */
    init.mode  = MD_GPIO_MODE_INPUT;
    init.odos  = MD_GPIO_PUSH_PULL;
    init.pupd  = MD_GPIO_PUSH_UP;
    init.podrv = MD_GPIO_OUT_DRIVE_6;
    init.nodrv = MD_GPIO_OUT_DRIVE_6;
    init.flt   = MD_GPIO_FILTER_DISABLE;
    init.type  = MD_GPIO_TYPE_CMOS;
    init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(LPUART0_RX_PORT, LPUART0_RX_PIN, &init);

    return;
}

/**
  * @brief  Initializate lpuart module.
  * @retval None
  */
void lpuart_init()
{
    md_lpuart_init_t init;
    /* clear md_lpuart_init_t structure */
    memset(&init, 0x0, sizeof(md_lpuart_init_t));
    /* Initialize LPUART */
    init.baud        = 9600;
    init.word_length = MD_LPUART_WORD_LENGTH_8B;
    init.stop_bits   = MD_LPUART_STOP_BITS_1;
    init.parity      = MD_LPUART_PARITY_NONE;
    init.mode        = MD_LPUART_MODE_UART;
    init.fctl        = MD_LPUART_FLOW_CTL_DISABLE;
    md_lpuart_init(LPUART0, &init);
}

/**
  * @brief  Initializate pin of leds.
  * @retval None
  */
void led_pin_init()
{
    md_gpio_init_t init;

    init.mode  = MD_GPIO_MODE_OUTPUT;
    init.odos  = MD_GPIO_PUSH_PULL;
    init.pupd  = MD_GPIO_PUSH_UP;
    init.podrv = MD_GPIO_OUT_DRIVE_6;
    init.nodrv = MD_GPIO_OUT_DRIVE_6;
    init.flt   = MD_GPIO_FILTER_DISABLE;
    init.type  = MD_GPIO_TYPE_CMOS;
    init.func  = MD_GPIO_FUNC_1;

    md_gpio_init(LED1_PORT, LED1_PIN, &init);
    md_gpio_init(LED2_PORT, LED2_PIN, &init);
    md_gpio_set_pin_high(LED1_PORT, LED1_PIN);
    md_gpio_set_pin_high(LED2_PORT, LED2_PIN);
}
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    BaseType_t retval;

    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC_6, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    /* Enable all peripherals clock */
    md_cmu_perh_clock_config(MD_CMU_PERH_ALL, ENABLE);

    /* Set LP16T0 clock source(LOSC) */
    SYSCFG_UNLOCK();
    md_cmu_set_lpuart0_clock_source(0x6);
    SYSCFG_LOCK();
    __enable_irq();

    /* Initialize LPUART */
    lpuart_pin_init();
    lpuart_init();
    /* Initialize LED */
    led_pin_init();

    printf("ES32H040x FreeRTOS demo\r\nSysCoreClock=%d\r\n", md_system_clock);
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
    }
}

int fputc(int ch, FILE *fp)
{
    UNUSED(fp);
    while (!md_lpuart_get_state_txemp(LPUART0));
    md_lpuart_send_data(LPUART0, (uint8_t)ch);
    while (md_lpuart_get_state_txemp(LPUART0));
    return 0;
}

/**
  * @brief  App task creator
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create LED1 task */
    retval = xTaskCreate(task_led1, "LED1_Task", 128, NULL, 1, &g_task_led1_handle);

    if (pdPASS == retval)
        printf("Task LED1 Created\r\n");

    /* Create LED2 task*/
    retval = xTaskCreate(task_led2, "LED2_Task", 128, NULL, 2, &g_task_led2_handle);

    if (pdPASS == retval)
        printf("Task LED2 Created\r\n");

    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  LED1 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_led1(void *param)
{
    uint32_t val = 0U;

    while (1)
    {
        /* Waiting for notify bit1=0x02 */
        if (xTaskNotifyWait(0, BIT_ALL, &val, portMAX_DELAY) == pdTRUE)
        {
            if ((val & BIT(1)) == BIT(1))
            {
                printf("Task1 receive notify %d\r\n", val);
                md_gpio_toggle_pin_output(LED1_PORT, LED1_PIN);
            }
        }
    }
}

/**
  * @brief  LED2 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_led2(void *param)
{
    uint32_t val = 0U;

    for (;;)
    {
        if (xTaskNotifyWait(0, BIT_ALL, &val, portMAX_DELAY) == pdTRUE)
        {
            /* Waiting for notify bit0 and bit2 = 0x05 */
            if ((val & (BIT(0) | BIT(2))) == (BIT(0) | BIT(2)))
            {
                printf("Task2 receive notify %d\r\n", val);
                md_gpio_toggle_pin_output(LED2_PORT, LED2_PIN);
            }
        }
    }
}

/**
  * @brief  Notify task1 and task2 to toggle LED (called from SysTick)
  * @param  param Parameter passed when created
  * @retval None
  */
void callback_from_irq()
{
    static uint32_t tick = 0;
    BaseType_t woken1 = pdFALSE, woken2 = pdFALSE;

    /* Noify task1 every 400 ticks */
    if (tick % 400 == 0)
        xTaskGenericNotifyFromISR(g_task_led1_handle, 0, BIT(1), eSetBits, NULL, &woken1);

    /* Noify task2 every 1000 ticks */
    if (tick % 1000 == 0)
        xTaskGenericNotifyFromISR(g_task_led2_handle, 0, BIT(0) | BIT(2), eSetBits, NULL, &woken2);

    tick++;
    portYIELD_FROM_ISR((woken1 || woken2));
}
/**
  * @}
  */
/**
  * @}
  */
