/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Aug 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *          12 Aug 2021     shiwa           FreeRTOS Demo
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "bsp_uart.h"
#include "bsp_led.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "bsp_uart.h"
#include "bsp_led.h"


/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

uint32_t g_sys_core_clk = 24000000U;
uint32_t g_sys_sleep_clk = 2000000U;

/* Private Variables--------------------------------------------------------- */

static TaskHandle_t s_task_app_handle = NULL;
static TaskHandle_t s_task_led1_handle = NULL;

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

static void task_app(void *);

static void task_led1(void *pvParameters);

static void bsp_init(void);

/* Private Function---------------------------------------------------------- */
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    BaseType_t retval;

    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_96M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 96000000);
    g_sys_core_clk = 96000000;
    md_init_1ms_tick();
    bsp_init();

    md_gpio_set_pin_low(GPIOF, MD_GPIO_PIN_1);
    md_delay_1ms(1000);
    md_gpio_set_pin_high(GPIOF, MD_GPIO_PIN_1);
    printf("ES32F3696 FreeRTOS Demo\r\n");
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &s_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @brief  App task creator
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create LED1 task */
    retval = xTaskCreate(task_led1, "LED1_Task", 128, NULL, 2, &s_task_led1_handle);

    if (pdPASS == retval)
        printf("Task LED1 Created\r\n");

    /* Delete self */
    vTaskDelete(s_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  LED1 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_led1(void *param)
{
    uint32_t i = 0U;

    while (1)
    {
        printf("Task Running, count %4d\r\n", i);

        /* Do some hard task here */
        md_delay_1ms(50);

        vTaskDelay(500);

        vTaskDelay(500);

        i++;
    }
}


/**
  * @brief  Setup all GPIO pins
  * @retval None
  */
static void config_gpio()
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode  = MD_GPIO_MODE_CLOSE;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(GPIOA, 0xFFFF, &gpio_init);
    md_gpio_init(GPIOB, 0xFFFF, &gpio_init);
    md_gpio_init(GPIOC, 0xFFFF, &gpio_init);
    md_gpio_init(GPIOD, 0xFFFF, &gpio_init);
    md_gpio_init(GPIOE, 0xFFFF, &gpio_init);
    md_gpio_init(GPIOF, 0xFFFF, &gpio_init);


}

/**
  * @brief  Initialize all peripheral
  * @retval None
  */
static void bsp_init(void)
{
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    md_init_1ms_tick();

    config_gpio();

    /* LED & KEY*/
    init_led();

    /* UART */
    init_uart();

}

/**
  * @brief  Pre Sleep Processing
  * @param  expected_idle_time Expected time in idle state
  * @retval None
  */
void pre_sleep_processing(uint32_t *expected_idle_time)
{
    /* Called by the kernel before it places the MCU into a sleep mode.

    NOTE:  Additional actions can be taken here to get the power consumption
    even lower.  For example, peripherals can be turned off here, and then back
    on again in the post sleep processing function.  For maximum power saving
    ensure all unused pins are in their lowest power state. */

    /*
      (*expected_idle_time) is set to 0 to indicate that PreSleepProcessing contains
      its own wait for interrupt or wait for event instruction and so the kernel vPortSuppressTicksAndSleep
      function does not need to execute the wfi instruction
    */
    *expected_idle_time = 0;

    /* Disable all peripherals to get the power consumption lower. */
    md_cmu_disable_perh_all();

    /*Enter to sleep Mode using the md function with WFI instruction*/
    md_pmu_sleep();
}

/**
  * @brief  Post Sleep Processing
  * @param  expected_idle_time Not used
  * @retval None
  */
void post_sleep_processing(uint32_t *expected_idle_time)
{
    /* Called by the kernel when the MCU exits a sleep mode */

    /* Avoid compiler warnings about the unused parameter. */
    (void) expected_idle_time;

    md_cmu_enable_perh_all();
}

/**
  * @}
  */
/**
  * @}
  */
