/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Aug 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *          12 Aug 2021     shiwa           FreeRTOS Demo
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "bsp_uart.h"
#include "bsp_led.h"


/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

uint32_t g_sys_core_clk = 24000000U;

/* Private Variables--------------------------------------------------------- */

static TaskHandle_t s_task_app_handle = NULL;
static TaskHandle_t s_task_led1_handle = NULL;
static TaskHandle_t s_task_led2_handle = NULL;
static TaskHandle_t s_task_notify_gen_handle = NULL;
static EventGroupHandle_t s_event_test = NULL;

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

static void task_app(void *);

static void task_led1(void *pvParameters);
static void task_led2(void *pvParameters);
static void task_notify_gen(void *pvParameters);

static void bsp_init(void);

/* Private Function---------------------------------------------------------- */

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    BaseType_t retval;

    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    g_sys_core_clk = 72000000;
    md_init_1ms_tick();

    bsp_init();
    printf("ES32F3696 FreeRTOS Demo\r\n");
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &s_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @brief  App task creator
  * @param  parameter: Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create event group */
    s_event_test = xEventGroupCreate();

    if (s_event_test != NULL)
        printf("Create Event OK\r\n");

    /* Create LED1 task */
    retval = xTaskCreate(task_led1, "LED1_Task", 128, NULL, 1, &s_task_led1_handle);

    if (pdPASS == retval)
        printf("Task LED1 Created\r\n");

    /* Create LED2 task*/
    retval = xTaskCreate(task_led2, "LED2_Task", 128, NULL, 2, &s_task_led2_handle);

    if (pdPASS == retval)
        printf("Task LED2 Created\r\n");

    /* Create Signal Gen task*/
    retval = xTaskCreate(task_notify_gen, "Notify_Task", 128, NULL, 3, &s_task_notify_gen_handle);

    if (pdPASS == retval)
        printf("Task Notify Gen Created\r\n");

    /* Delete self */
    vTaskDelete(s_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  LED1 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_led1(void *param)
{
    while (1)
    {
        /* Waiting for notify bit1 */
        xEventGroupWaitBits(s_event_test, BIT(0), pdTRUE, pdTRUE, portMAX_DELAY);
        md_gpio_toggle_pin_output(LED_PORT, LED1_PIN);
        printf("Task1 get events ok!\r\n");
    }
}

/**
  * @brief  LED2 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_led2(void *param)
{

    for (;;)
    {
        xEventGroupWaitBits(s_event_test, (BIT(0) | BIT(1)), pdTRUE, pdTRUE, portMAX_DELAY);
        md_gpio_toggle_pin_output(LED_PORT, LED2_PIN);
        printf("Task2 get events ok!\r\n");
    }
}

/**
  * @brief  Generate event notify to task1 and task2
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_notify_gen(void *param)
{
    uint32_t val = 0U;
    uint32_t i = 0U;
    const uint32_t EV_SEQ[] = {BIT(0), BIT(1), BIT(2), (BIT(0) | BIT(1)), (BIT(1) | BIT(2)), (BIT(0) | BIT(1) | BIT(2)) };

    for (;;)
    {
        /* Generate notify sequence 001 010 100 011 110 111 */
        for (i = 0; i < 6; i++)
        {
            val = EV_SEQ[i];
            printf("Signal gen %u\r\n", val);
            xEventGroupSetBits(s_event_test, val);

            vTaskDelay(500);
        }

        val = 0;
    }
}

/**
  * @brief  Initialize all peripheral
  * @retval None
  */
static void bsp_init(void)
{
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* LED */
    init_led();

    /* UART */
    init_uart();

}
/**
  * @}
  */
/**
  * @}
  */
