/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Aug 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *          12 Aug 2021     shiwa           FreeRTOS Demo
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "bsp_uart.h"
#include "bsp_led.h"


/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

uint32_t g_sys_core_clk = 24000000U;

/* Private Variables--------------------------------------------------------- */

static TaskHandle_t s_task_app_handle = NULL;
static TaskHandle_t s_task_led1_handle = NULL;
static TimerHandle_t s_timer_test;
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

static void task_app(void *);

static void task_led1(void *pvParameters);
static void timer_callback(TimerHandle_t xTimer);

static void bsp_init(void);

/* Private Function---------------------------------------------------------- */

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    BaseType_t retval;

    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    g_sys_core_clk = 72000000;
    md_init_1ms_tick();

    bsp_init();
    printf("ES32F3696 FreeRTOS Demo\r\n");
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &s_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @brief  App task creator
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create and start timer for every 1000ms */
    s_timer_test = xTimerCreate("Test", 1000, pdTRUE, 0, timer_callback);

    if (s_timer_test != NULL)
        printf("Timer created\r\n");

    xTimerStart(s_timer_test, 0);

    /* Create LED1 task */
    retval = xTaskCreate(task_led1, "LED1_Task", 128, NULL, 2, &s_task_led1_handle);

    if (pdPASS == retval)
        printf("Task LED1 Created\r\n");

    /* Delete self */
    vTaskDelete(s_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  LED1 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_led1(void *param)
{
    uint32_t i = 0U;

    while (1)
    {
        printf("LED1_Task Running, count %d\r\n", i);

        md_gpio_set_pin_low(GPIOF, MD_GPIO_PIN_0);
        vTaskDelay(500);

        md_gpio_set_pin_high(GPIOF, MD_GPIO_PIN_0);
        vTaskDelay(500);

        i++;
    }
}


/**
  * @brief  Timer callback
  * @param  param Parameter passed when created
  * @retval None
  */
static void timer_callback(TimerHandle_t xTimer)
{
    printf("Timer called\r\n");
    md_gpio_toggle_pin_output(LED_PORT, LED2_PIN);
}

/**
  * @brief  Initialize all peripheral
  * @retval None
  */
static void bsp_init(void)
{
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* LED */
    init_led();

    /* UART */
    init_uart();

}
/**
  * @}
  */
/**
  * @}
  */
