/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Aug 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *          12 Aug 2021     shiwa           FreeRTOS Demo
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "bsp_uart.h"
#include "bsp_led.h"


/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

uint32_t g_sys_core_clk = 24000000U;

/* Private Variables--------------------------------------------------------- */

static TaskHandle_t s_task_app_handle = NULL;
static TaskHandle_t s_task_high_pri_handle = NULL;
static TaskHandle_t s_task_medium_pri_handle = NULL;
static TaskHandle_t s_task_low_pri_handle = NULL;
static uint32_t s_high_pri_cycle = 0U;
static uint32_t s_medium_pri_cycle = 0U;
static uint32_t s_low_pri_cycle = 0U;
static SemaphoreHandle_t s_mutex_test;
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

static void task_app(void *);

static void task_high_pri(void *pvParameters);
static void task_medium_pri(void *pvParameters);
static void task_low_pri(void *pvParameters);

static void bsp_init(void);

/* Private Function---------------------------------------------------------- */;

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    BaseType_t retval;

    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    g_sys_core_clk = 72000000;
    md_init_1ms_tick();

    bsp_init();
    printf("ES32F3696 FreeRTOS Demo\r\n");
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &s_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @brief  App task creator
  * @param  parameter: Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();
    /* Create mutx or binary semaphore */
#if 1
    s_mutex_test = xSemaphoreCreateMutex();
#else
    s_mutex_test = xSemaphoreCreateBinary();
    xSemaphoreGive(s_mutex_test);
#endif /* Set to 1 to use mutex, 0 to use binary semaphore */

    if (s_mutex_test != NULL)
        printf("Create mutex OK\r\n");

    /* Create high priority task */
    retval = xTaskCreate(task_high_pri, "HighPriTask", 128, NULL, 5, &s_task_high_pri_handle);

    if (pdPASS == retval)
        printf("Task high priority Created\r\n");

    /* Create medium priority task*/
    retval = xTaskCreate(task_medium_pri, "MediumPriTask", 128, NULL, 4, &s_task_medium_pri_handle);

    if (pdPASS == retval)
        printf("Task medium priority Created\r\n");

    /* Create low priority task*/
    retval = xTaskCreate(task_low_pri, "LowPriTask", 128, NULL, 3, &s_task_low_pri_handle);

    if (pdPASS == retval)
        printf("Task low priority Created\r\n");

    /* Delete self */
    vTaskDelete(s_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  High priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_high_pri(void *param)
{
    while (1)
    {
        /* Get mutex */
        if (xSemaphoreTake(s_mutex_test, portMAX_DELAY) != pdTRUE)
        {
            printf("ERR:task_high_pri take failed\r\n");
        }

        printf("Mutex Take:  1\r\n");
        vTaskDelay(1000);

        if (xSemaphoreGive(s_mutex_test) != pdTRUE)
        {
            /* Toggle LED3 to indicate error */
            printf("ERR:task_high_pri give failed\r\n");
        }

        /* Keep count of the number of cycles this thread has performed */
        s_high_pri_cycle++;

        vTaskSuspend(NULL);
    }
}

/**
  * @brief  Medium priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_medium_pri(void *param)
{
    while (1)
    {
        if (xSemaphoreTake(s_mutex_test, portMAX_DELAY) == pdTRUE)
        {

            printf("Mutex Take:    2\r\n");
            vTaskDelay(1000);

            if (eTaskGetState(s_task_high_pri_handle) != eSuspended)
            {
                /* Did not expect to execute until the high priority thread was
                suspended.
                Toggle LED3 to indicate error */
                printf("ERR:task_high_pri should be suspended!\r\n");
            }
            else
            {
                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(s_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_medium_pri give failed");
                }

                vTaskSuspend(NULL);
            }
        }
        else
        {
            /* We should not leave the osMutexWait() function
            until the mutex was obtained.
            Toggle LED3 to indicate error */
            printf("ERR:task_medium_pri cant take failed\r\n");
        }

        /* The High and Medium priority threads should be in lock step */
        if (s_high_pri_cycle != (s_medium_pri_cycle + 1))
        {
            printf("Lock step failed\r\n");
        }

        /* Print a message to indicate medium task is running */
        printf("Task2 Run :    2\r\n");

        /* Keep count of the number of cycles this task has performed so a
        stall can be detected */
        s_medium_pri_cycle++;
        vTaskDelay(500);
    }
}

/**
  * @brief  Low priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_low_pri(void *param)
{
    while (1)
    {
        /* Keep attempting to obtain the mutex.  We should only obtain it when
        the medium-priority thread has suspended itself, which in turn should only
        happen when the high-priority thread is also suspended */
        if (xSemaphoreTake(s_mutex_test, portMAX_DELAY) == pdTRUE)
        {
            printf("Mutex Take:      3\r\n");
            vTaskDelay(1000);

            /* Is the haigh and medium-priority threads suspended? */
            if ((eTaskGetState(s_task_high_pri_handle) != eSuspended) || (eTaskGetState(s_task_medium_pri_handle) != eSuspended))
            {
                printf("ERR:task_high_pri and task_medium_pri should be suspended!\r\n");
            }
            else
            {
                /* Keep count of the number of cycles this task has performed
                so a stall can be detected */
                s_low_pri_cycle++;

                /* We can resume the other tasks here even though they have a
                higher priority than the this thread. When they execute they
                will attempt to obtain the mutex but fail because the low-priority
                thread is still the mutex holder.  this thread will then inherit
                the higher priority.  The medium-priority thread will block indefinitely
                when it attempts to obtain the mutex, the high-priority thread will only
                block for a fixed period and an error will be latched if the
                high-priority thread has not returned the mutex by the time this
                fixed period has expired */
                /* If binary semaphore is used, you will see "Task 2 run" before we give
                mutex to high priority task. Otherwise, Task 2 would not run before high
                priority task run*/
                vTaskResume(s_task_high_pri_handle);
                vTaskResume(s_task_medium_pri_handle);

                /* The other two tasks should now have executed and no longer
                be suspended */
                if ((eTaskGetState(s_task_high_pri_handle) == eSuspended) || (eTaskGetState(s_task_medium_pri_handle) == eSuspended))
                {
                    printf("ERR:task_high_pri and task_medium_pri should NOT be suspended!\r\n");
                }

                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(s_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_low_pri give failed");
                }
            }
        }
    }
}

/**
  * @brief  Initialize all peripheral
  * @retval None
  */
static void bsp_init(void)
{
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* LED */
    init_led();

    /* UART */
    init_uart();

}
/**
  * @}
  */
/**
  * @}
  */
