/**
  *********************************************************************************
  *
  * @file    bsp_uart.c
  * @brief   UART for debug printf
  *
  * @version V1.0
  * @date    12 Aug 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Aug 2021     shiwa           The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


/* Includes ----------------------------------------------------------------- */

#include "bsp_uart.h"

/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
static uint32_t s_uart_enabled = 0;
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */


/**
  * @brief  Initialize UART for debug output/input
  * @retval None
  */
void init_uart(void)
{
    md_gpio_init_t x;
    md_uart_init_t h_uart;

    md_gpio_init_struct(&x);
    /* Initialize tx pin*/
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(M_UART_TX_PORT, M_UART_TX_PIN, &x);

    /* Initialize rx pin*/
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(M_UART_RX_PORT, M_UART_RX_PIN, &x);

    /* Initialize UART */
    md_uart_init_struct(&h_uart);

    h_uart.baud        = M_UART_BAUD;
    h_uart.word_length = MD_UART_WORD_LENGTH_8B;
    h_uart.stop_bits   = MD_UART_STOP_BITS_1;
    h_uart.parity      = MD_UART_PARITY_NONE;
    h_uart.fctl        = MD_UART_FLOW_CTL_DISABLE;
    h_uart.mode        = MD_UART_MODE;
    md_uart_init(M_UART, &h_uart);

    md_uart_set_tx_threshold(M_UART, MD_UART_TXFIFO_EMPTY);
    md_uart_set_rx_threshold(M_UART, MD_UART_RXFIFO_1BYTE);

    md_uart_set_rec_timeout(M_UART, 0xFF);

    md_mcu_irq_config(M_UART_IRQ, 3, 3, ENABLE);

    s_uart_enabled = 1;
}

/**
  * @brief  Redirect fputc to UART0
  * @param  ch Character to print
  * @param  f Not used
  * @retval None
  */
int fputc(int ch, FILE *f)
{
    if (!s_uart_enabled)
        return 0;

    /* Send one byte */
    md_uart_set_send_data8(M_UART, (ch & 0xFF));

    /* Wait for complete */
    while (md_uart_is_active_flag_tfth(M_UART) == RESET);

    return (ch);
}

/**
  * @brief  Redirect fgetc to UART0
  * @param  f Not used
  * @retval None
  */
int fgetc(FILE *f)
{
    /* Wait for data */
    while (md_uart_is_active_flag_rfth(M_UART) == RESET);

    /* Recv one byte */
    return (int)md_uart_recv_data8(M_UART);
}
