/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "bsp_uart.h"
#include "bsp_led.h"


/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

uint32_t g_sys_core_clk = 24000000U;

/* Private Variables--------------------------------------------------------- */

static TaskHandle_t s_task_app_handle = NULL;
static TaskHandle_t s_task_send_handle = NULL;
static TaskHandle_t s_task_recv_handle = NULL;

static QueueHandle_t s_queue_test;
static uint32_t s_producer_val = 0, s_consumer_val = 0;
static volatile uint32_t s_producer_err = 0, s_consumer_err = 0;
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

static void task_app(void *);

static void task_send(void *pvParameters);
static void task_recv(void *pvParameters);

static void bsp_init(void);

/* Private Function---------------------------------------------------------- */
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    BaseType_t retval;

    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    g_sys_core_clk = 72000000;
    md_init_1ms_tick();

    bsp_init();
    printf("ES32F3696 FreeRTOS Demo\r\n");
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &s_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @brief  App task creator
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create Queue length = 5, item_size = 4 */
    s_queue_test = xQueueCreate(5, 4);

    if (s_queue_test != NULL)
        printf("Create Test_Queue OK\r\n");

    /* Create LED1 task */
    retval = xTaskCreate(task_send, "Send_Task", 128, NULL, 2, &s_task_send_handle);

    if (pdPASS == retval)
        printf("Task Send Created\r\n");

    /* Create LED2 task*/
    retval = xTaskCreate(task_recv, "Recv_Task", 128, NULL, 3, &s_task_recv_handle);

    if (pdPASS == retval)
        printf("Task Recv Created\r\n");

    /* Delete self */
    vTaskDelete(s_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  Send task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_send(void *param)
{
    while (1)
    {
        /* Send message to queue */
        if (xQueueSend(s_queue_test, &s_producer_val, 3000) != pdTRUE)
        {
            /* Error */
            ++s_producer_err;
            printf("Send error count:%d\r\n", s_producer_err);
        }
        else
        {
            /* Update producer count */
            ++s_producer_val;

            /* Toggle LED1 and print send count */
            md_gpio_toggle_pin_output(LED_PORT, LED1_PIN);
            printf("Send count:%d\r\n", s_producer_val);

            vTaskDelay(1000);
        }
    }
}

/**
  * @brief  Recv task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_recv(void *param)
{
    uint32_t val = 0U;

    while (1)
    {
        /* Receive message from queue */
        if (xQueueReceive(s_queue_test, &val, 800) == pdTRUE)
        {
            /* Check if value received data correct */
            if (val == s_consumer_val)
            {
                /* Update producer count */
                ++s_consumer_val;
                /* Toggle LED2 and print receive count */
                md_gpio_toggle_pin_output(LED_PORT, LED2_PIN);
                printf("Recv count:%d\r\n", s_consumer_val);
            }
            else
            {
                /* Error */
                ++s_consumer_err;
                printf("Recv error count:%d\r\n", s_consumer_err);
            }
        }
        else
        {
            printf("Recv nothing!\r\n");
        }
    }
}

/**
  * @brief  Initialize all peripheral
  * @retval None
  */
static void bsp_init(void)
{
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* LED */
    init_led();

    /* UART */
    init_uart();

}
/**
  * @}
  */
/**
  * @}
  */
