/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          22 Mar 2022     AE Team         Modify MD Driver
 *          10 Oct 2022     AE Team         FreeRTOS demo
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_PASS,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HOSCON | RCU_CON_HRCON | RCU_CON_PLL0ON | RCU_CON_HRC48ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
typedef struct my_mail
{
    uint32_t a;
    uint32_t b;
    uint8_t c;
} my_mail_t;
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
static volatile uint32_t g_producer_err = 0, g_consumer_err = 0;
static TaskHandle_t g_task_app_handle = NULL;
static TaskHandle_t g_task_send_handle = NULL;
static TaskHandle_t g_task_recv_handle = NULL;
static QueueHandle_t g_queue_test = NULL;
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

static void task_app(void *);
static void task_send(void *pvParameters);
static void task_recv(void *pvParameters);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    BaseType_t retval;
    
    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    bsp_led_init();
    bsp_key_init();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();
    
    printf("ES32F0283 FreeRTOS demo\r\nSysCoreClock=%d\r\n",SystemFrequency_SysClk);
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /* Tx FIFO empty */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @brief  App task creator
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create Queue length = 5, item_size = sizeof(my_mail_t) */
    g_queue_test = xQueueCreate(5, sizeof(my_mail_t));

    if (g_queue_test != NULL)
        printf("Create Test_Queue OK\r\n");

    /* Create sender task */
    retval = xTaskCreate(task_send, "Send_Task", 128, NULL, 2, &g_task_send_handle);

    if (pdPASS == retval)
        printf("Task Send Created\r\n");

    /* Create receiver task*/
    retval = xTaskCreate(task_recv, "Recv_Task", 128, NULL, 3, &g_task_recv_handle);

    if (pdPASS == retval)
        printf("Task Recv Created\r\n");

    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  Send task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_send(void *param)
{
    my_mail_t msg;
    msg.a = 0;
    msg.b = 0;
    msg.c = 0;

    while (1)
    {
        /* Send message to queue */
        if (xQueueSend(g_queue_test, &msg, 3000) != pdTRUE)
        {
            /* Error */
            ++g_producer_err;
            printf("Send error count:%d\r\n", g_producer_err);
        }
        else
        {
            /* Update producer count */
            msg.a += 1;
            msg.b += 2;
            msg.c += 3;
            /* Toggle LED1 and print send count */
            LED_TOGGLE(LED_PORT, LED1_PIN);
            printf("Send    :a=%u,b=%u,c=%u\r\n", msg.a, msg.b, msg.c);
            vTaskDelay(1000);
        }
    }
}

/**
  * @brief  Recv task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_recv(void *param)
{
    uint32_t val_a = 0U, val_b = 0U;
    uint8_t val_c = 0U;
    my_mail_t msg;

    while (1)
    {
        /* Receive message from queue */
        if (xQueueReceive(g_queue_test, &msg, 1000) == pdTRUE)
        {
            /* Check if value received data correct */
            if (msg.a == val_a && msg.b == val_b && msg.c == val_c)
            {
                /* Toggle LED2 and print receive count */
                LED_TOGGLE(LED_PORT, LED2_PIN);
                printf("Received:a=%u,b=%u,c=%u\r\n", msg.a, msg.b, msg.c);

                /* Update producer count */
                val_a += 1;
                val_b += 2;
                val_c += 3;
            }
            else
            {
                /* Error */
                ++g_consumer_err;
                printf("RecvError:a=%u,b=%u,c=%u\r\n", msg.a, msg.b, msg.c);
            }
        }
        else
        {
            printf("Recv nothing!\r\n");
        }
    }
}
/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


