/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          22 Mar 2022     AE Team         Modify MD Driver
 *          10 Oct 2022     AE Team         FreeRTOS demo
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_PASS,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HOSCON | RCU_CON_HRCON | RCU_CON_PLL0ON | RCU_CON_HRC48ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
static TaskHandle_t g_task_app_handle = NULL;
static TaskHandle_t g_task_high_pri_handle = NULL;
static TaskHandle_t g_task_medium_pri_handle = NULL;
static TaskHandle_t g_task_low_pri_handle = NULL;
static uint32_t g_high_pri_cycle = 0U;
static uint32_t g_medium_pri_cycle = 0U;
static uint32_t g_low_pri_cycle = 0U;
static SemaphoreHandle_t g_mutex_test;
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

static void task_app(void *);
static void task_high_pri(void *pvParameters);
static void task_medium_pri(void *pvParameters);
static void task_low_pri(void *pvParameters);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    BaseType_t retval;
    
    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    bsp_led_init();
    bsp_key_init();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();
    
    printf("ES32F0283 FreeRTOS demo\r\nSysCoreClock=%d\r\n",SystemFrequency_SysClk);
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /* Tx FIFO empty */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @brief  App task creator
  * @param  parameter: Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();
    /* Create mutex or binary semaphore */
#if 1
    g_mutex_test = xSemaphoreCreateMutex();
#else
    g_mutex_test = xSemaphoreCreateBinary();
    xSemaphoreGive(g_mutex_test);
#endif /* Set to 1 to use mutex, 0 to use binary semaphore */

    if (g_mutex_test != NULL)
        printf("Create mutex OK\r\n");

    /* Create high priority task */
    retval = xTaskCreate(task_high_pri, "HighPriTask", 128, NULL, 5, &g_task_high_pri_handle);

    if (pdPASS == retval)
        printf("Task high priority Created\r\n");

    /* Create medium priority task*/
    retval = xTaskCreate(task_medium_pri, "MediumPriTask", 128, NULL, 4, &g_task_medium_pri_handle);

    if (pdPASS == retval)
        printf("Task medium priority Created\r\n");

    /* Create low priority task*/
    retval = xTaskCreate(task_low_pri, "LowPriTask", 128, NULL, 3, &g_task_low_pri_handle);

    if (pdPASS == retval)
        printf("Task low priority Created\r\n");

    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  High priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_high_pri(void *param)
{
    while (1)
    {
        /* Get mutex */
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) != pdTRUE)
        {
            printf("ERR:task_high_pri take failed\r\n");
        }

        printf("Mutex Take:  1\r\n");
        vTaskDelay(1000);

        if (xSemaphoreGive(g_mutex_test) != pdTRUE)
        {
            /* Toggle LED3 to indicate error */
            printf("ERR:task_high_pri give failed\r\n");
        }

        /* Keep count of the number of cycles this thread has performed */
        g_high_pri_cycle++;

        vTaskSuspend(NULL);
    }
}

/**
  * @brief  Medium priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_medium_pri(void *param)
{
    while (1)
    {
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) == pdTRUE)
        {

            printf("Mutex Take:    2\r\n");
            vTaskDelay(1000);

            if (eTaskGetState(g_task_high_pri_handle) != eSuspended)
            {
                /* Did not expect to execute until the high priority thread was
                suspended.
                Toggle LED3 to indicate error */
                printf("ERR:task_high_pri should be suspended!\r\n");
            }
            else
            {
                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(g_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_medium_pri give failed");
                }

                vTaskSuspend(NULL);
            }
        }
        else
        {
            /* We should not leave the osMutexWait() function
            until the mutex was obtained.
            Toggle LED3 to indicate error */
            printf("ERR:task_medium_pri cant take failed\r\n");
        }

        /* The High and Medium priority threads should be in lock step */
        if (g_high_pri_cycle != (g_medium_pri_cycle + 1))
        {
            printf("Lock step failed\r\n");
        }

        /* Print a message to indicate medium task is running */
        printf("Task2 Run :    2\r\n");

        /* Keep count of the number of cycles this task has performed so a
        stall can be detected */
        g_medium_pri_cycle++;
        vTaskDelay(500);
    }
}

/**
  * @brief  Low priority task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_low_pri(void *param)
{
    while (1)
    {
        /* Keep attempting to obtain the mutex.  We should only obtain it when
        the medium-priority thread has suspended itself, which in turn should only
        happen when the high-priority thread is also suspended */
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) == pdTRUE)
        {
            printf("Mutex Take:      3\r\n");
            vTaskDelay(1000);

            /* Is the haigh and medium-priority threads suspended? */
            if ((eTaskGetState(g_task_high_pri_handle) != eSuspended) || (eTaskGetState(g_task_medium_pri_handle) != eSuspended))
            {
                printf("ERR:task_high_pri and task_medium_pri should be suspended!\r\n");
            }
            else
            {
                /* Keep count of the number of cycles this task has performed
                so a stall can be detected */
                g_low_pri_cycle++;

                /* We can resume the other tasks here even though they have a
                higher priority than the this thread. When they execute they
                will attempt to obtain the mutex but fail because the low-priority
                thread is still the mutex holder.  this thread will then inherit
                the higher priority.  The medium-priority thread will block indefinitely
                when it attempts to obtain the mutex, the high-priority thread will only
                block for a fixed period and an error will be latched if the
                high-priority thread has not returned the mutex by the time this
                fixed period has expired */
                /* If binary semaphore is used, you will see "Task 2 run" before we give
                mutex to high priority task. Otherwise, Task 2 would not run before high
                priority task run*/
                vTaskResume(g_task_high_pri_handle);
                vTaskResume(g_task_medium_pri_handle);

                /* The other two tasks should now have executed and no longer
                be suspended */
                if ((eTaskGetState(g_task_high_pri_handle) == eSuspended) || (eTaskGetState(g_task_medium_pri_handle) == eSuspended))
                {
                    printf("ERR:task_high_pri and task_medium_pri should NOT be suspended!\r\n");
                }

                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(g_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_low_pri give failed");
                }
            }
        }
    }
}
/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


