/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          22 Mar 2022     AE Team         Modify MD Driver
 *          10 Oct 2022     AE Team         FreeRTOS demo
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_PASS,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HOSCON | RCU_CON_HRCON | RCU_CON_PLL0ON | RCU_CON_HRC48ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
static TaskHandle_t g_task_app_handle = NULL;
static TaskHandle_t g_task_send_handle = NULL;
static TaskHandle_t g_task_recv_handle = NULL;
static SemaphoreHandle_t g_sema_test;
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

static void task_app(void *);
static void task_send(void *pvParameters);
static void task_recv(void *pvParameters);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    BaseType_t retval;
    
    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    bsp_led_init();
    bsp_key_init();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();
    
    printf("ES32F0283 FreeRTOS demo\r\nSysCoreClock=%d\r\n",SystemFrequency_SysClk);
    /* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should not run to here */
    while (1)
    {
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /* Tx FIFO empty */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @brief  App task creator
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create semaphore */
    g_sema_test = xSemaphoreCreateBinary();

    if (g_sema_test != NULL)
        printf("Create Test_Queue OK\r\n");

    xSemaphoreGive(g_sema_test);

    /* Create send task */
    retval = xTaskCreate(task_send, "Send_Task", 128, NULL, 4, &g_task_send_handle);

    if (pdPASS == retval)
        printf("Task send created\r\n");

    /* Create recv task*/
    retval = xTaskCreate(task_recv, "Recv_Task", 128, NULL, 3, &g_task_recv_handle);

    if (pdPASS == retval)
        printf("Task recv created\r\n");

    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  Send task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_send(void *param)
{
    uint32_t count = 0U;

    for (;;)
    {

        /* Try to obtain the semaphore */
        if (xSemaphoreTake(g_sema_test, 1000) == pdTRUE)
        {
            printf("Task1 get Semaphore\r\n");
            count = xTaskGetTickCount() + 5000;

            /* Toggle LED1 every 200 ms for 5 seconds */
            while (count > xTaskGetTickCount())
            {
                /* Toggle LED1 */
                LED_TOGGLE(LED_PORT,LED1_PIN);
                /* Delay 200 ms */
                vTaskDelay(200);
            }

            /* Turn off LED1*/
            md_gpio_set_pin_high(LED_PORT, LED1_PIN);
            /* Release the semaphore */
            xSemaphoreGive(g_sema_test);

            /* Suspend ourseleves to execute thread 2 (lower priority)  */
            vTaskSuspend(NULL);
        }
        else
        {
            printf("Task1 failed to get Semaphore\r\n");
        }
    }
}

/**
  * @brief  Recv task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_recv(void *param)
{
    uint32_t count = 0U;

    for (;;)
    {
        /* Try to obtain the semaphore */
        if (xSemaphoreTake(g_sema_test, 1000) == pdTRUE)
        {
            printf("Task2 get Semaphore\r\n");
            /* Resume Thread 1 (higher priority)*/
            vTaskResume(g_task_send_handle);

            count = xTaskGetTickCount() + 5000;

            /* Toggle LED2 every 200 ms for 5 seconds*/
            while (count > xTaskGetTickCount())
            {
                LED_TOGGLE(LED_PORT,LED2_PIN);

                vTaskDelay(200);
            }

            /* Turn off LED2 */
            md_gpio_set_pin_high(LED_PORT, LED2_PIN);

            /* Release the semaphore to unblock Thread 1 (higher priority)  */
            xSemaphoreGive(g_sema_test);
        }
        else
        {
            printf("Task2 failed to get Semaphore\r\n");
        }

    }
}
/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


