/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    07 Dec 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          07 Dec 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "micro_sd.h"
#include "string.h"

/* Private Macros ------------------------------------------------------------ */
#define DELAY_CNT       30
#define EDIT_BLK_SIZE   512
#define CID_SIZ         16
#define CSD_SIZE        16

/* flash information */

/* function */
#define SD_CS_SET() ald_gpio_write_pin(SD_PORT, SD_CS_PIN, 1)
#define SD_CS_RST() ald_gpio_write_pin(SD_PORT, SD_CS_PIN, 0)



/* Private Variables --------------------------------------------------------- */
static spi_handle_t s_spi_handle;
static uint8_t s_data_buf[20];
static uint8_t s_sd_type;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void spi_pin_init(void)
{
    gpio_init_t gpio_config;

    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_PUSH_PULL;
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.type = GPIO_TYPE_TTL;
    gpio_config.func = GPIO_FUNC_1;

    ald_gpio_init(SD_PORT, SD_CS_PIN, &gpio_config);  /*Initialize the selected pin*/
    SD_CS_SET();   /*Choose the output high, release of Flash*/

    gpio_config.func = GPIO_FUNC_4;
    ald_gpio_init(SD_PORT, SD_SCL_PIN, &gpio_config);  /*Initialize the clock output pin*/

    gpio_config.func = GPIO_FUNC_4;
    ald_gpio_init(SD_PORT, SD_MOSI_PIN, &gpio_config);  /*Initialize the MOSI pin*/

    gpio_config.mode = GPIO_MODE_INPUT;
    gpio_config.func = GPIO_FUNC_4;
    ald_gpio_init(SD_PORT, SD_MISO_PIN, &gpio_config);  /*Initializes the MISO pins*/

    return;
}

/**
  * @brief spi init function
  * @retval None.
  */
static ald_status_t spi_init(void)
{
    spi_pin_init();

    s_spi_handle.perh           = SPI1;                /*Using SPI1 */
    s_spi_handle.init.mode      = SPI_MODE_MASTER;     /*SPI host mode */
    s_spi_handle.init.baud      = SPI_BAUD_64;         /*clock / 64 */
    s_spi_handle.init.data_size = SPI_DATA_SIZE_8;     /*8 bit pattern */
    s_spi_handle.init.polarity  = SPI_CPOL_HIGH;       /* Free high level */
    s_spi_handle.init.phase     = SPI_CPHA_SECOND;     /* The second edge receiving data */
    s_spi_handle.init.first_bit = SPI_FIRSTBIT_MSB;    /* Send the MSB first */
    s_spi_handle.init.dir       = SPI_DIRECTION_2LINES;
    s_spi_handle.init.ss_en     = DISABLE;
    s_spi_handle.init.crc_calc  = DISABLE;

    if (ald_spi_init(&s_spi_handle) == OK)  /*According to the parameter initialization SPI peripherals*/
        return OK;
    else
        return ERROR;
}

/**
  * @brief  Delay.
  * @param  delay: times of delay.
  * @retval None.
  */
static void delay(uint32_t delay)
{
    while (delay--);
}

/**
  * @brief  Send a command to card.
  * @param  cmd: Type of command.
  * @param  arg: Parameter of command.
  * @param  crc: CRC of command.
  * @retval return OK if function execute successfuly, otherwise, reutrn ERROR.
  */
static int sd_send_cmd(sd_card_cmd_t cmd, uint32_t arg, uint8_t crc)
{
    uint8_t buf[6];
    uint8_t ret, i;
    uint16_t retry = 10;
    int r_flag = 0;

    SD_CS_SET();
    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    SD_CS_RST();
    delay(DELAY_CNT);

    buf[0] = cmd | 0x40;
    buf[1] = (arg >> 24) & 0xFF;
    buf[2] = (arg >> 16) & 0xFF;
    buf[3] = (arg >> 8) & 0xFF;
    buf[4] = arg & 0xFF;
    buf[5] = crc;

    for (i = 0; i < 6; ++i)
        ald_spi_send_byte_fast(&s_spi_handle, buf[i]);

    do
    {
        ret = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
    }
    while ((ret == 0xFF) && (retry--));

    return ret;
}

/**
  * @brief  Write data to card.
  * @param  buf: Pointer to data.
  * @param  cmd: Type of command
  * @retval Card's status.
  */
static int sd_write_block(uint8_t *buf, uint8_t cmd)
{
    uint8_t ret = 0;
    uint16_t retry = 10;
    int r_flag = 0;

    do
    {
        ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    }
    while (--retry);

    ald_spi_send_byte_fast(&s_spi_handle, 0xFE);

    for (retry = 0; retry < 512; ++retry)
    {
        ald_spi_send_byte_fast(&s_spi_handle, buf[retry]);
    }

    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    delay(DELAY_CNT * 4);

    ret = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);

    if ((ret & 0x1f) != 0x5)
    {
        ret = 1;
        goto end;
    }

    retry = 0xFFFF;

    do
    {
        ret = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
    }
    while ((ret != 0xFF) && (--retry));

    ret = retry == 0 ? 2 : 0;

end:
    SD_CS_SET();
    delay(DELAY_CNT);
    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    delay(DELAY_CNT);

    return ret;
}

/**
  * @brief  Read data from card.
  * @param  buf: Pointer to data.
  * @param  len: Length of data.
  * @retval Card's status.
  */
static int sd_read_data(uint8_t *buf, uint16_t len)
{
    uint8_t ret;
    uint16_t retry = 0xFFFF;
    int r_flag = 0;

    do
    {
        ret = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
    }
    while ((ret != 0xFE) && (--retry));

    if (!retry)
        return -1;

    while (len--)
    {
        *buf++ = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
    }

    /* Read CRC */
    s_data_buf[18] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
    s_data_buf[19] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);

    SD_CS_SET();
    delay(DELAY_CNT);
    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    delay(DELAY_CNT);

    return 0;
}

/**
  * @brief  Erase the blocks in the card
  * @param  start_blk: starting of blocks.
  * @param  end_blk: ending of blocks.
  * @retval Status:
  *          -  0 Success
  *          - -1 Failed
  */
static int sd_blocks_erase(uint32_t start_blk, uint32_t end_blk)
{
    uint32_t retry = 10;
    uint8_t ret;

    do
    {
        ret = sd_send_cmd(SD_CMD32, start_blk, 0x1);
    }
    while ((ret != 0) && (--retry));

    if (ret != 0)
        goto end;

    retry = 10;

    do
    {
        ret = sd_send_cmd(SD_CMD33, end_blk, 0x1);
    }
    while ((ret != 0) && (--retry));

    if (ret != 0)
        goto end;

    retry = 10;

    do
    {
        ret = sd_send_cmd(SD_CMD38, 0, 0x1);
    }
    while ((ret != 0) && (--retry));

    if (ret != 0)
        goto end;

    SD_CS_SET();
    delay(DELAY_CNT);
    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    delay(DELAY_CNT);
    return 0;

end:
    SD_CS_SET();
    delay(DELAY_CNT);
    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    delay(DELAY_CNT);
    return -1;
}

/**
  * @brief  Write data to the block in the card
  * @param  buf: Pointer to the data.
  * @param  idx_blk: Index of block in the card.
  * @retval Status:
  *          -  0 Success
  *          - -1 Failed
  */
static int micro_sd_write_block(const uint8_t *buf, uint32_t idx_blk)
{
    uint8_t ret, retry = 10;
    int result = 1;

    do
    {
        ret = sd_send_cmd(SD_CMD24, idx_blk, 0x1);
    }
    while ((ret != 0) && (--retry));

    if (ret == 0x0)
        result = sd_write_block((uint8_t *)buf, 0xFE);

    return result;
}

/**
  * @brief  Read data from the sector in the card
  * @param  buf: Pointer to the data.
  * @param  start_sector: Index of sector in the card.
  * @param  sec_num:      Number of sectors to read.
  * @retval Status:
  *          -  0 Success
  *          - -1 Failed
  */
static int micro_sd_read_block(uint8_t *buf, uint32_t idx_blk)
{
    uint8_t ret, retry = 10;

    do
    {
        ret = sd_send_cmd(SD_CMD17, idx_blk, 0x1);
    }
    while ((ret != 0) && (--retry));

    if (ret == 0x0)
        ret = sd_read_data(buf, EDIT_BLK_SIZE);

    return ret;
}

/**
  * @brief  init micro sd card
  * @param  None
  * @retval Status:
  *          -  0 Success
  *          - -1 Failed
  */
static int micro_sd_init(void)
{
    uint32_t retry = 2, i;
    uint8_t ret;
    int r_flag = 0;

    SD_CS_SET();

    for (i = 0; i < 20; ++i)
        ald_spi_send_byte_fast(&s_spi_handle, 0xFF);

    delay(DELAY_CNT);

    /* Send CMD0 */
    do
    {
        ret = sd_send_cmd(SD_CMD0, 0, 0x95);
    }
    while ((ret != 1) && (--retry));

    /* Send CMD8 */
    ret = sd_send_cmd(SD_CMD8, 0x1AA, 0x87);

    if (ret == 0x1)
    {
        s_data_buf[0] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
        s_data_buf[1] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
        s_data_buf[2] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
        s_data_buf[3] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
    }

    if ((s_data_buf[2] != 0x1) || (s_data_buf[3] != 0xAA))
        return -2;

    /* Send CMD55+ACMD41 */
    retry = 500;

    do
    {
        ret = sd_send_cmd(SD_CMD55, 0, 0x1);

        if (ret == 1)
            ret = sd_send_cmd(SD_ACMD41, 0x40000000, 0x1);
    }
    while ((ret != 0) && (--retry));

    if (ret != 0)
        return -3;

    /* Send CMD58 */
    ret = sd_send_cmd(SD_CMD58, 0, 0x1);

    if (ret == 0x0)
    {
        s_data_buf[10] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
        s_data_buf[11] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
        s_data_buf[12] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
        s_data_buf[13] = ald_spi_recv_byte_fast(&s_spi_handle, &r_flag);
    }

    if ((s_data_buf[10] & 0x40) == 0x40)
        s_sd_type = 4; /* V2-HC */

    SD_CS_SET();
    delay(DELAY_CNT);
    ald_spi_send_byte_fast(&s_spi_handle, 0xFF);
    delay(DELAY_CNT);
    SPI_DISABLE(&s_spi_handle);
    ald_spi_speed_config(&s_spi_handle, SPI_BAUD_8);
    SPI_ENABLE(&s_spi_handle);
    return 0;
}

/* Public Function ---------------------------------------------------------- */
/**
  * @brief  Get type of the card.
  * @retval type of micro sd card.
  */
uint32_t sd_get_type(void)
{
    return s_sd_type;
}

/**
  * @brief  Get CID of the card.
  * @param  buf: Pointer to buffer.
  * @retval Status.
  */
int sd_get_cid(uint8_t *buf)
{
    uint8_t ret;

    ret = sd_send_cmd(SD_CMD10, 0, 0x1);

    if (ret == 0x0)
        ret = sd_read_data(buf, CID_SIZ);

    return 0;
}

/**
  * @brief  Get size of the card.
  * @retval Number of blocks(512 byte).
  */
uint32_t sd_get_size(void)
{
    uint8_t ret, n;
    uint32_t size, capacity;

    ret = sd_send_cmd(SD_CMD9, 0, 0x1);

    if (ret == 0x0)
        ret = sd_read_data(s_data_buf, CSD_SIZE);

    /* SDHC */
    if ((s_data_buf[0] & 0xC0) == 0x40)
    {
        size = s_data_buf[9] + ((uint16_t)s_data_buf[8] << 8) + 1;
        capacity = (uint32_t)size << 10;
    }
    else     /* V1 */
    {
        n = (s_data_buf[5] & 15) + ((s_data_buf[10] & 128) >> 7) + ((s_data_buf[9] & 3) << 1) + 2;
        size = (s_data_buf[8] >> 6) + ((uint16_t)s_data_buf[7] << 2) + ((uint16_t)(s_data_buf[6] & 3) << 10) + 1;
        capacity = (uint32_t)size << (n - 9);
    }

    return capacity;
}

/**
  * @brief  micro sd card init function
  * @retval None.
  */
ald_status_t ll_micro_sd_init(void)
{
    memset(s_data_buf, 0xFF, 20);

    /* Initialization SPI IO */
    spi_pin_init();

    /* Initialization SPI peripherals */
    if (spi_init() != OK)
        return ERROR;

    if (micro_sd_init())
        return ERROR;

    return OK;
}

/**
  * @brief  micro sd card erase sector function
  * @retval None.
  */
ald_status_t ll_micro_sd_sector_erase(uint32_t start_sec, uint32_t end_sec)
{
    if (sd_blocks_erase(start_sec, end_sec))
        return ERROR;
    else
        return OK;
}

/**
  * @brief  micro sd card sectors read function
  * @retval None.
  */
ald_status_t ll_micro_sd_sector_read(uint8_t *str_buf, uint32_t sec, uint32_t sec_num)
{
    uint32_t i;

    for (i = 0; i < sec_num; i++)
    {
        if (micro_sd_read_block(str_buf + i * EDIT_BLK_SIZE, sec + i))
            return ERROR;
    }

    return OK;
}

/**
  * @brief  micro sd card sectors write function
  * @retval None.
  */
ald_status_t ll_micro_sd_sector_write(const uint8_t *src_buf, uint32_t sec, uint32_t sec_num)
{
    uint32_t i;

    for (i = 0; i < sec_num; i++)
    {
        if (micro_sd_write_block(src_buf + i * EDIT_BLK_SIZE, sec + i))
            return ERROR;
    }

    return OK;
}


/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
