/**********************************************************************************
 *
 * @file    nand.h
 * @brief   Header file for nand.c
 *
 * @date    12 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          12 Nov 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "sram_diskio.h"

/* Private Macros ------------------------------------------------------------ */
/* Block Size in Bytes */
#define SEC_SIZE            512
#define BLOCK_SIZE          2
#define DEV_SIZE            0x100000

/* Private variables ---------------------------------------------------------*/

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS sram_init(BYTE);
DSTATUS sram_stat(BYTE);
DRESULT sram_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT sram_write(BYTE lun, const BYTE *, DWORD, UINT);
#endif /* FF_FS_READONLY == 0 */

#if  _USE_IOCTL == 1
DRESULT sram_ioctl(BYTE, BYTE, void *);
#endif

/* Public Variables ---------------------------------------------------------- */
const Diskio_drvTypeDef sram_drv =
{
    sram_init,
    sram_stat,
    sram_read,

#if  _USE_WRITE
    sram_write,
#endif /* FF_FS_READONLY == 0 */

#if  _USE_IOCTL == 1
    sram_ioctl,
#endif
};

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS sram_init(BYTE lun)
{
    /* Configure the SRAM device */
    if (ll_ebisram_init())
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS sram_stat(BYTE lun)
{
    return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT sram_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    if (ll_ebisram_read(buff, (uint32_t)(sector), count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if  _USE_WRITE
DRESULT sram_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    if (ll_ebisram_write(buff, sector, count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}
#endif /* FF_FS_READONLY == 0 */

#if _USE_IOCTL == 1
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT sram_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = DEV_SIZE / SEC_SIZE;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = SEC_SIZE;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = BLOCK_SIZE;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            while (1);

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}
#endif

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
