/**********************************************************************************
 *
 * @file    nand.h
 * @brief   Header file for nand.c
 *
 * @date    12 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          12 Nov 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "nand_flash_diskio.h"

/* Private Macros ------------------------------------------------------------ */
/* Block Size in Bytes */
#define SEC_SIZE            0x0800
#define BLK_SIZE            0x0040                      /* 128K */
#define DEV_SIZE            0x4000000                   /* 64MB */

/* Private variables ---------------------------------------------------------*/
static uint8_t ctrl_buf[FF_MAX_SS] = {0};

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS nand_flash_init(BYTE);
DSTATUS nand_flash_stat(BYTE);
DRESULT nand_flash_read(BYTE, BYTE *, DWORD, UINT);
#if _USE_WRITE
    DRESULT nand_flash_write(BYTE, const BYTE *, DWORD, UINT);
#endif
#if _USE_IOCTL
DRESULT nand_flash_ioctl(BYTE, BYTE, void *);
#endif
DRESULT move_block(BYTE lun, BYTE *, DWORD sector, UINT count);
DRESULT content_check(BYTE lun, BYTE *, const BYTE *, DWORD sector, UINT count);

/* Public Variables ---------------------------------------------------------- */
const Diskio_drvTypeDef nand_flash_drv =
{
    nand_flash_init,
    nand_flash_stat,
    nand_flash_read,

#if _USE_WRITE
    nand_flash_write,
#endif /* FF_FS_READONLY == 0 */

#if _USE_IOCTL
    nand_flash_ioctl,
#endif
};

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS nand_flash_init(BYTE lun)
{
    /* Configure the NAND flash device */
    if (ll_nand_flash_init())
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS nand_flash_stat(BYTE lun)
{
    return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT nand_flash_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    if (ll_nand_flash_read(buff, (uint32_t)(sector), count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if  _USE_WRITE
DRESULT nand_flash_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    uint32_t i;

    /* content check */
    if (content_check(lun, ctrl_buf, buff, sector, count) == RES_OK)
        return RES_OK;

    /* clear target sectors */
    move_block(lun, ctrl_buf, sector, count);

    for (i = 0; i < count; i++)
    {
        /* write data to expected area */
        if (ll_nand_flash_write(buff, sector + i, 1))
        {
            return RES_ERROR;
        }

        /* verify the written data */
        if (ll_nand_flash_read(ctrl_buf, sector + i, 1))
        {
            return RES_ERROR;
        }

        if (memcmp((const void *)ctrl_buf, (const void *)buff, sizeof(ctrl_buf)))
            return RES_ERROR;
    }

    return RES_OK;
}
#endif /* FF_FS_READONLY == 0 */

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT nand_flash_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = DEV_SIZE / SEC_SIZE;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = SEC_SIZE;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = BLK_SIZE;
            res = RES_OK;
            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

#endif

/**
 * @brief  copy data to backup area
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT move_block(BYTE lun, BYTE *buf, DWORD sector, UINT count)
{
    uint32_t i, j, start_sec, end_sec, cluster_cnt;
    uint32_t bkp_cluster, bkp_start_sec, bkp_end_sec, clus_cnt, cnt;
    uint32_t len;
    uint32_t *ptr = (void *)0;

    /* calculate cluster scope in unit of sector */
    ll_block_scope_get(sector, &start_sec, &end_sec);

    /* get block number which need moved to backup area*/
    if (sector + count - 1 > end_sec)
    {
        /* nearby blocks */
        clus_cnt = 1 + ((sector + count - end_sec) / BLK_SIZE + (((sector + count - end_sec) % BLK_SIZE) ? 0 : 1));
    }
    else
    {
        /* in the same block */
        clus_cnt = 1;
    }

    /* get total cluster number */
    cluster_cnt = DEV_SIZE / (BLK_SIZE * SEC_SIZE);

    /* find a free area as the backup area, cluster number-i, cluster start sector number-j */
    for (i = 0; i < cluster_cnt; i++)
    {
        bkp_cluster = cluster_cnt - i - 1;
        bkp_start_sec = bkp_cluster * BLK_SIZE;
        bkp_end_sec = bkp_start_sec + BLK_SIZE - 1;

        /* verify if no data in current sector */
        for (j = 0; j < BLK_SIZE; j++)
        {
            if (ll_nand_flash_read(buf, bkp_start_sec + j, 1))
            {
                return RES_ERROR;
            }
            
            /* due to memmory media is always alligned as 4bytes */
            ptr = (uint32_t *)buf;
            for (len = 0; len < SEC_SIZE; len += 4)
            {
                if (*ptr != 0xFFFFFFFF) 
                    break;
                ptr++;
            }
            
            if (len < SEC_SIZE)
                break;
        }

        if (j == BLK_SIZE)
            break;
    }

    /* backup data in cycle */
    for (cnt = 0 ; cnt < clus_cnt; cnt++)
    {
        /* copy data in current block and write to backup area */
        for (i = start_sec + cnt * BLK_SIZE; i <= end_sec + cnt * BLK_SIZE; i++)
        {
            /* if it's target sector, then ignore */
            if (i >= sector && i < sector + count)
                continue;

            /* otherwise, read data from current sector */
            if (ll_nand_flash_read(buf, i, 1))
            {
                return RES_ERROR;
            }

            /* write data to backup area */
            if (ll_nand_flash_write(buf, bkp_start_sec + i - start_sec + cnt * BLK_SIZE, 1))
            {
                return RES_ERROR;
            }
        }

        /* erase current block */
        if (ll_nand_flash_erase(start_sec + cnt * BLK_SIZE, end_sec + cnt * BLK_SIZE))
            return RES_ERROR;

        /* write back data */
        for (i = start_sec + cnt * BLK_SIZE; i <= end_sec + cnt * BLK_SIZE; i++)
        {
            /* if it's target sector, then ignore */
            if (i >= sector && i < sector + count)
                continue;

            /* otherwise, read data from backup area */
            if (ll_nand_flash_read(buf, bkp_start_sec + i - start_sec + cnt * BLK_SIZE, 1))
            {
                return RES_ERROR;
            }

            /* write data back */
            if (ll_nand_flash_write(buf, i, 1))
            {
                return RES_ERROR;
            }
        }

        /* erase backup area */
        if (ll_nand_flash_erase(bkp_start_sec, bkp_end_sec))
            return RES_ERROR;
    }

    return RES_OK;
}

/**
 * @brief  compare content.
 * @param  cmd: Control code.
 * @param  lun.
 * @param  rbuf: buffer to store data.
 * @param  buf: buffer under compare.
 * @param  sector.
 * @param  count.
 * @retval DRESULT: RES_OK or RES_ERROR.
 */
DRESULT content_check(BYTE lun, BYTE *rbuf, const BYTE *buf, DWORD sector, UINT count)
{
    uint32_t i;
    for (i = 0; i < count; i++)
    {
        if (ll_nand_flash_read(rbuf, sector + i, 1))
        {
            return RES_ERROR;
        }
        if (memcmp(rbuf + i * 0x800, buf + i * 0x800, 0x800))	//NAND_PAGE_SIZE
        {
            return RES_ERROR;
        }
    }
    return RES_OK;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
