/**********************************************************************************
 *
 * @file    nand_flash.c
 * @brief   Main file for DEMO
 *
 * @date    12 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          12 Nov 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "nand_flash.h"

/* Private Macros ------------------------------------------------------------ */
#define     EBI_NAND_DATA_PORTn         8
#define     EBI_NAND_DATA_PINn          8
#define     EBI_D0_PORT                 GPIOD
#define     EBI_D0_PIN                  GPIO_PIN_14
#define     EBI_D1_PORT                 GPIOD
#define     EBI_D1_PIN                  GPIO_PIN_15
#define     EBI_D2_PORT                 GPIOD
#define     EBI_D2_PIN                  GPIO_PIN_0
#define     EBI_D3_PORT                 GPIOD
#define     EBI_D3_PIN                  GPIO_PIN_1
#define     EBI_D4_PORT                 GPIOE
#define     EBI_D4_PIN                  GPIO_PIN_7
#define     EBI_D5_PORT                 GPIOE
#define     EBI_D5_PIN                  GPIO_PIN_8
#define     EBI_D6_PORT                 GPIOE
#define     EBI_D6_PIN                  GPIO_PIN_9
#define     EBI_D7_PORT                 GPIOE
#define     EBI_D7_PIN                  GPIO_PIN_10
#define     EBI_NCE2_PORT               GPIOD
#define     EBI_NCE2_PIN                GPIO_PIN_7
#define     EBI_NOE_PORT                GPIOD
#define     EBI_NOE_PIN                 GPIO_PIN_4
#define     EBI_NWE_PORT                GPIOD
#define     EBI_NWE_PIN                 GPIO_PIN_5
#define     EBI_NWAIT_PORT              GPIOD
#define     EBI_NWAIT_PIN               GPIO_PIN_6
#define     EBI_CLE_PORT                GPIOD
#define     EBI_CLE_PIN                 GPIO_PIN_11
#define     EBI_ALE_PORT                GPIOD
#define     EBI_ALE_PIN                 GPIO_PIN_12

#define NAND_PAGE_SIZE        0x0800  /* 2048 bytes per page w/o Spare Area */
#define NAND_BLOCK_SIZE       0x0040  /* 64x2048 bytes pages per block */
#define NAND_PLANE_SIZE       0x0400  /* 1024 Block per plane */
#define NAND_SPARE_AREA_SIZE  0x0040  /* last 64 bytes as spare area */
#define NAND_BLOCK_NBR        0x0400  /* 1 planes of 1024 block */
#define NAND_PLANE_NBR        1       /* 1 device of 1 planes */
#define BUFFER_SIZE           0x800   /* (NAND_PAGE_SIZE * NAND_PLANE_NBR) */

#define STP_OK  0
#define STP_NOK 1

/* Public Variables ---------------------------------------------------------- */
nand_handle_t  g_nand_handle;
nand_address_t g_nand_addr;

/* Private Constants --------------------------------------------------------- */
GPIO_TypeDef *EBI_DATA_PORT[EBI_NAND_DATA_PORTn] =
{
    EBI_D0_PORT, EBI_D1_PORT,
    EBI_D2_PORT, EBI_D3_PORT,
    EBI_D4_PORT, EBI_D5_PORT,
    EBI_D6_PORT, EBI_D7_PORT
};
const uint16_t EBI_DATA_PIN[EBI_NAND_DATA_PINn] =
{
    EBI_D0_PIN, EBI_D1_PIN,
    EBI_D2_PIN, EBI_D3_PIN,
    EBI_D4_PIN, EBI_D5_PIN,
    EBI_D6_PIN, EBI_D7_PIN
};

/* Private Variables--------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Translate logical address into a phy one.
  * @param  sec: sec number in total nand flash.
  * @param  n_addr: NAND addr.
  * @retval Status
  */
static void ll_nand_addr_get(uint32_t sec, nand_address_t *n_addr)
{
    uint32_t addr = sec * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE);
    n_addr->page  = (addr % (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE);
    n_addr->block = (addr % (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
    n_addr->plane = addr / (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));

    return;
}

/**
  * @brief  Nand pin init.
  * @param  None
  * @retval None
  */
static void ll_nand_flash_pins_init(void)
{
    uint8_t i;
    gpio_init_t gpio_config;

    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /* Data pin init */
    for (i = 0; i < EBI_NAND_DATA_PINn; ++i)
    {
        ald_gpio_init(EBI_DATA_PORT[i], EBI_DATA_PIN[i], &gpio_config);
    }

    /* Control pin init */
    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /* Output enable pin init */
    ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &gpio_config);
    /* Write enable init EBI_NWE*/
    ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &gpio_config);
    /* Chip select pin init EBI_NCE2 */
    ald_gpio_init(EBI_NCE2_PORT, EBI_NCE2_PIN, &gpio_config);
    /* EBI_CLE init */
    ald_gpio_init(EBI_CLE_PORT, EBI_CLE_PIN, &gpio_config);
    /* Address Latch pin init EBI_ALE */
    ald_gpio_init(EBI_ALE_PORT, EBI_ALE_PIN, &gpio_config);
    /* Ready/Busy pin init */
    gpio_config.mode = GPIO_MODE_INPUT;
    gpio_config.func = GPIO_FUNC_7;
    ald_gpio_init(EBI_NWAIT_PORT, EBI_NWAIT_PIN, &gpio_config);

    return;
}

/**
  * @brief  Low-level driver initialization for NAND FLASH
  * @retval Status.
  */
uint8_t ll_nand_flash_init(void)
{
    /* Initializa structure for timing */
    ald_ebi_nand_timing_t timing;

    g_nand_handle.instance          = EBI_NAND_DEVICE;
    g_nand_handle.init.bank         = EBI_NAND_BANK2;
    g_nand_handle.init.wait         = EBI_NAND_WAIT_FEATURE_ENABLE;
    g_nand_handle.init.width        = EBI_NAND_MEM_BUS_WIDTH_8;
    g_nand_handle.init.ecc          = EBI_NAND_ECC_ENABLE;
    g_nand_handle.init.size         = EBI_NAND_ECC_PAGE_SIZE_2048BYTE;
    g_nand_handle.init.cle_time     = 0x0;
    g_nand_handle.init.ale_time     = 0x0;

    g_nand_handle.config.page_size  = NAND_PAGE_SIZE;
    g_nand_handle.config.spare_size = NAND_SPARE_AREA_SIZE;
    g_nand_handle.config.block_size = NAND_BLOCK_SIZE;
    g_nand_handle.config.block_nbr  = NAND_BLOCK_NBR;
    g_nand_handle.config.plane_nbr  = NAND_PLANE_NBR;
    g_nand_handle.config.plane_size = NAND_PLANE_SIZE;

    /*限制条件为：nand flash 手册中的相关参数 + 芯片外设限制。
    HCLK2最大频率72M，需求<= 13ns的不考虑*/
    timing.hiz_time    = (((ald_cmu_get_hclk2_clock()) * 7) / 50000000); //(hclk2_clk*140)/1000000000;
    timing.wait_time   = ((timing.hiz_time) * 4) / 3;
    timing.hold_time   = timing.wait_time / 2;
    timing.time        = timing.hold_time;

    if ((timing.hiz_time - timing.time) < 2)
    {
        timing.wait_time++;
        timing.hold_time++;
    }

    /*将实际时序转为寄存器值+时序寄存器范围限制*/
    timing.hiz_time     = (timing.hiz_time < 0x01) ?  0x00 : (timing.hiz_time - 1); /*MEMHIZTx/ATTHIZTx   reg范围 0x00--0xFE  HCLK周期 = reg+1 */
    timing.hiz_time     = (timing.hiz_time > 0xFE) ?  0xFE : timing.hiz_time;
    timing.hold_time    = (timing.hold_time < 0x01) ? 0x01 : timing.hold_time;      /*MEMHOLDx/ATTHOLDx   reg范围 0x01--0xFE  HCLK周期 = reg   */
    timing.hold_time    = (timing.hold_time > 0xFE) ? 0xFE : timing.hold_time;
    timing.wait_time    = (timing.wait_time < 0x01) ? 0x00 : (timing.wait_time - 1); /*MEMWAITx/ATTWAITx   reg范围 0x01--0xFF  HCLK周期 = reg+1 */
    timing.wait_time    = (timing.wait_time > 0xFF) ? 0xFF : timing.wait_time;
    timing.time         = (timing.time < 0x01) ?      0x00 : (timing.time - 1);     /*MEMSETUPx/ATTSETUPx reg范围 0x00--0xFE  HCLK周期 = reg+1 */
    timing.time         = (timing.time > 0xFE) ?      0xFE : timing.time;

    /* EBI pins initialization */
    ll_nand_flash_pins_init();

    /* NAND Flash initialization */
    if (ald_nand_init(&g_nand_handle, &timing, &timing))
        return STP_NOK;

    if (ald_nand_reset(&g_nand_handle))
        return STP_NOK;

    return STP_OK;
}

/**
  * @brief  Read data from nand flash via low-level driver
  * @retval Status.
  */
uint8_t ll_nand_flash_read(uint8_t *str_buf, uint32_t sec, uint32_t sec_num)
{
    ll_nand_addr_get(sec, &g_nand_addr);

    if (ald_nand_read_page_8b(&g_nand_handle, &g_nand_addr, str_buf, sec_num))
        return STP_NOK;

    return STP_OK;
}

/**
  * @brief  Write data to nand flash via low-level driver
  * @retval Status.
  */
uint8_t ll_nand_flash_write(const uint8_t *src_buf, uint32_t sec, uint32_t sec_num)
{
    ll_nand_addr_get(sec, &g_nand_addr);

    if (ald_nand_write_page_8b(&g_nand_handle, &g_nand_addr, (uint8_t *)src_buf, sec_num))
        return STP_NOK;

    return STP_OK;
}

/**
  * @brief  Erase block via low-level driver
  * @retval Status.
  */
uint8_t ll_nand_flash_erase(uint32_t start_sec, uint32_t end_sec)
{
    for (; start_sec <= end_sec; start_sec += NAND_BLOCK_SIZE)
    {
        ll_nand_addr_get(start_sec, &g_nand_addr);

        if (ald_nand_erase_block(&g_nand_handle, &g_nand_addr))
            return STP_NOK;
    }

    return STP_OK;
}
/**
  * @brief  calculate current block scope.
  * @param  sec: target sec number in total nand flash.
  * @param  start_sec: start sec number in the block of target sector.
  * @param  end_sec: end sec number in the block of target sector.
  * @retval none
  */
void ll_block_scope_get(uint32_t sec, uint32_t *start_sec, uint32_t *end_sec)
{
    *start_sec = (sec / NAND_BLOCK_SIZE) * NAND_BLOCK_SIZE;
    *end_sec = *start_sec + NAND_BLOCK_SIZE - 1;

    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
