/**********************************************************************************
 *
 * @file    main.c
 * @brief   main file for UART-Printf sample project
 *
 * @version V1.0
 * @date    2021.07.05
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2021.03.04      lvzd            the 1st version
 *          2021.07.05      biyq            the 2nd version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <math.h>
#include "main.h"
#include "md_gpio.h"

/* Private Macros ----------------------------------------------------------- */

/* Private Variables -------------------------------------------------------- */
uint8_t g_tx_buf[500];

/* Public Variables --------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private Function Prototypes ---------------------------------------------- */
void     uart_pin_init(void);
uint32_t data_initialize(uint8_t *, uint8_t *, uint32_t);

/* Public Function --------------------------------------------------------- */


/**
  * @brief  main function
  * @param  None
  * @retval None
  */
int main()
{
    uint32_t i, j, length;
    int32_t data2send[2];
    md_uart_init_t uart_init;

    /* 配置系统时钟 */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* 初始化SysTick中断 */
    md_init_1ms_tick();

    /* 使能所有外设时钟 */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* 初始化UART引脚 */
    uart_pin_init();

    /* 配置UART通信参数 */
    memset(&uart_init, 0x0, sizeof(md_uart_init_t));
    uart_init.baud        = 115200;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    md_uart_init(UART0, &uart_init);

    i = 0;

    while (1)
    {
        data2send[0] = (int32_t)(100 * sin((2 * 3.1415926 / 2047) * i)); /* 正弦数据 */
        data2send[1] = (int32_t)(100 * cos((2 * 3.1415926 / 2047) * i)); /* 余弦数据 */

        /* 按照ES-DAP-Viewer的UARTPrint传输协议要求, 格式化传输数据，获取待发送数据的总长度
           格式化后的数据暂存在全局变量 g_tx_buf[20] 里，注意数组长度不要越界 */
        length = data_initialize(g_tx_buf, (uint8_t *)data2send, 2 * 4);

        /* 发送格式化后的数据 */
        for (j = 0; j < length; j++)
        {
            md_uart_set_send_data8(UART0, g_tx_buf[j]);

            while (RESET == md_uart_is_active_it_tbc(UART0));

            md_uart_clear_it_tbc(UART0);
        }

        i++;
    }
}

/* Private Function --------------------------------------------------------- */
/**
  * @brief  Initializate UART pin
  * @param  None
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize tx pin：PB10 */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_TTL;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &gpio_config);

    /* Initialize rx pin：PB11 */
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_TTL;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &gpio_config);

    return;
}

/**
  * @brief  格式化用户数据，使之符合传输协议
  * @param  srcdata-保存用户数据的首地址；
  *         desdata-保存处理好的数据的内存首地址(其内存大小至少大于前者5字节)；
  *         len-用户数据长度(单位:字节)
  * @retval 返回数据帧的总长度
  */
uint32_t data_initialize(uint8_t *desdata, uint8_t *srcdata, uint32_t len)
{
    uint32_t i;

    /* 帧头 */
    desdata[0] = 0xAA;
    desdata[1] = 0xAA;

    /* 功能码 */
    desdata[2] = 0xF1;

    /* 数据长度 */
    desdata[3] = len;

    /* 用户数据 */
    memcpy(desdata + 4, srcdata, len);

    /* 累加校验和 */
    desdata[len + 4] = 0U;

    for (i = 0; i < len + 4; i++)
    {
        /* 累加校验和, 高位舍去, 只保留低8位 */
        desdata[len + 4] += desdata[i];
    }

    return (len + 5);
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
