/**********************************************************************************
 *
 * @file    stdio_uart.c
 * @brief   redirect stdio to uart
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
 
/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include <stdio.h>
#include "stdio_uart.h"
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define UART_BUF_LEN 256

/* Private Variables --------------------------------------------------------- */
uint8_t g_rx_buf[UART_BUF_LEN];
uint32_t g_rx_ptr = 0;
uint32_t g_rx_read = 0;
uint32_t g_rx_len = 0;

/* Public Variables ---------------------------------------------------------- */
uart_handle_t g_uart_init;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void init_uart_pin(void)
{
    gpio_init_t gpio_init;
    memset(&gpio_init, 0x00, sizeof(gpio_init_t));

    /* Initialize tx pin */
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin */
    gpio_init.mode = GPIO_MODE_INPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);

    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(uart_handle_t *arg)
{
    if (g_rx_ptr==UART_BUF_LEN-1)
    {
        g_rx_ptr=0;
    }
    else
    {
        g_rx_ptr++;
    }
    if (g_rx_len<UART_BUF_LEN)
    {
        g_rx_len++;
    }
    ald_uart_recv_by_it(&g_uart_init,g_rx_buf+g_rx_ptr,1);
    return;
}
void init_uart()
{
    init_uart_pin();
    
    memset(&g_uart_init, 0x00, sizeof(uart_handle_t));
    /* Initialize uart */
    g_uart_init.perh             = UART_X;
    g_uart_init.init.baud        = 115200;
    g_uart_init.init.word_length = UART_WORD_LENGTH_8B;
    g_uart_init.init.stop_bits   = UART_STOP_BITS_1;
    g_uart_init.init.parity      = UART_PARITY_NONE;
    g_uart_init.init.mode        = UART_MODE_UART;
    g_uart_init.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    g_uart_init.rx_cplt_cbk      = uart_recv_complete;
    g_uart_init.tx_cplt_cbk      = NULL;
    ald_uart_init(&g_uart_init);

    ald_uart_rx_fifo_config(&g_uart_init, UART_RXFIFO_1BYTE);
    ald_uart_tx_fifo_config(&g_uart_init, UART_TXFIFO_EMPTY);
}

int fputc(int chr, FILE *file)
{
    uint8_t c = chr;
    ald_uart_send(&g_uart_init, &c, 1, 1000);
    return chr;
}

int uart_send_str(const char *str, uint32_t len, uint32_t timeout)
{
    return ald_uart_send(&g_uart_init, (uint8_t *)str, len, timeout);
}
int uart_recv_str(char *str, uint32_t len, uint32_t timeout)
{
    return ald_uart_recv(&g_uart_init, (uint8_t *)str, len, timeout);
}
int uart_peak_ch()
{
    char ch=0;
    if (g_rx_len>0)
    {
        g_rx_len--;
        ch=g_rx_buf[g_rx_read];
        g_rx_read++;
        if (g_rx_read>=UART_BUF_LEN)
        {
            g_rx_read++;
        }
    }
    return ch;
}
