/**********************************************************************************
 *
 * @file    .h
 * @brief   Head file of boot flash module.
 *
 * @version V1.0
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          31 Dec 2019     AE Team         The first version
 *          20 Mar 2023     AE Team         Used as shell download/send
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#ifndef __XMODEM_FRAME_H__
#define __XMODEM_FRAME_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include "ald_conf.h"
#include "list.h"

#define XMODE_STX   0x02    /**< header of 1k-xmode  frames */
#define XMODE_EOT   0x04    /**< frame end byte */
#define XMODE_ACK   0x06    /**< frame ACK byte */
#define XMODE_NACK  0x15    /**< frame NACK byte */
#define XMODE_CAN   0x18    /**< revocation frame transmission */
#define XMODE_EOF   0x1A    /**< padding data */
#define XMODE_CRC   0x43    /**< CRC checkout frame */
#define TIME_OUT    3000    /**< timeout(ms) */
#define TRY_CNT     10  /**< repeat try number */

/**
  * @brief frame error state
  */
typedef enum
{
    FRAME_SUCCESS         = 0x01,   /**< Frame transmission sucess */
    FRAME_INDEX_ERROR     = 0x02,   /**< Frame index error */
    FRAME_RECEIVE_TIMEOUT = 0x04,   /**< Timeout error */
    FRAME_VERIFY_ERROR    = 0x08,   /**< Frame verigy error */
    UART_SEND_ERROR       = 0x10,   /**< UART send error */
    DATA_CBK_ERROR        = 0x20,   /**< Data callback returns an error */
    FRAME_SEND_TIMEOUT    = 0x40,   /**< Data callback returns an error */
    FRAME_SEND_NACK       = 0x40,   /**< Data callback returns an error */
} frames_state_t;

/**
  * @brief Command
  */
typedef enum
{
    XMOD_TX_CMD_START = XMODE_CRC,  /**< Start transfer */
    XMOD_TX_CMD_TERM  = XMODE_CAN,  /**< Terminal transfer */
    XMOD_TX_CMD_ACK   = XMODE_ACK,  /**< Acknowledgement */
    XMOD_TX_CMD_NCK   = XMODE_NACK, /**< Non-Acknowledgement */
} xmod_tx_cmd_t;

typedef int (*xm_frame_data_cbk_t)(void*usrdata,int idx, uint8_t*data, uint32_t crc);
/**
  * @brief Frame environment structure
  */
typedef struct xm_frame_env_s
{
    uint16_t idx;           /**< Current index of receive frame */
    uint8_t cnt;            /**< Number of repetitions */
    frames_state_t state;       /**< State */
    uint8_t soh;            /**< Start of header */
    uint8_t data[1029];     /**< Cache of the frame */
    xm_frame_data_cbk_t data_cbk;
    void*usr_data;
} boot_frame_env_t;
/**
  * @}
  */

/** @addtogroup Frame_Public_Variables
  * @{
  */
extern boot_frame_env_t g_frame_env;
/**
  * @}
  */

/** @addtogroup Frame_Public_Functions
  * @{
  */
extern const char* xm_get_info(void);
extern void xm_frame_init(xm_frame_data_cbk_t data_cbk,void*usrdata);
extern int  xm_download(void);
extern int  xm_send(void);

#define XMODE_1K

/*Implements these */
extern int uart_send_str(const char*str,uint32_t len,uint32_t timeout);
extern int uart_recv_str(char*str,uint32_t len,uint32_t timeout);
extern uint32_t calc_crc_ccitt(uint8_t*data,uint32_t initial,uint32_t len);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
extern "C"
}
#endif /* __cplusplus */
#endif /* __XMODEM_FRAME_H__ */
