#ifndef __LIST_H__
#define __LIST_H__


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


typedef struct list_head
{
    struct list_head *next;
    struct list_head *prev;
} list_head_t;

#define INIT_LIST_HEAD(ptr) do {    \
        (ptr)->next = (ptr);        \
        (ptr)->prev = (ptr);        \
    } while (0)


/**
  * @brief Insert a new entry between two entries.
  */

static __INLINE__ void __list_add(list_head_t *_new, list_head_t *_prev, list_head_t *_next)
{
    _next->prev = _new;
    _new->next  = _next;
    _new->prev  = _prev;
    _prev->next = _new;
}

/**
  * @brief Insert a new entry after the specified head.
  */
static __INLINE__ void list_add(list_head_t *_new, list_head_t *_head)
{
    __list_add(_new, _head, _head->next);
}

/**
  * @brief Insert a new entry before the specified head.
  */
static __INLINE__ void list_add_tail(list_head_t *_new, list_head_t *_head)
{
    __list_add(_new, _head->prev, _head);
}


/**
  * @brief Delete a list entry.
  */
static __INLINE__ void __list_del(list_head_t *_prev, list_head_t *_next)
{
    _next->prev = _prev;
    _prev->next = _next;
}

/**
  * @brief Deletes entry from list.
  */
static __INLINE__ void list_del(list_head_t *_entry)
{
    __list_del(_entry->prev, _entry->next);
    _entry->next = 0;
    _entry->prev = 0;
}

/**
  * @brief Tests whether a list is empty
  */
static __INLINE__ int list_empty(list_head_t *_head)
{
    return _head->next == _head;
}

/**
  * @brief Tests a head has one entry
  */
static __INLINE__ int list_empty_one(list_head_t *_head)
{
    return _head->next == _head->prev;
}

/**
 * @brief Join two lists
 */
static __INLINE__ void list_splice(list_head_t *list, list_head_t *head)
{
    list_head_t *first = list->next;

    if (first != list)
    {
        list_head_t *last = list->prev;
        list_head_t *at   = head->next;

        first->prev = head;
        head->next  = first;
        last->next  = at;
        at->prev    = last;
    }
}

/**
  * list_entry - get the struct for this entry
  */
#define list_entry(ptr, type, member) \
    ((type *)(uint32_t)((char *)(ptr) - (unsigned long)(&((type *)0)->member)))
#define offset_of(type, member) \
    ((uint32_t)&(((type *)0)->member))

/**
 * @brief Iterate over a list
 */
#define list_for_each(pos, head) \
    for (pos = (head)->next; pos != (head); \
            pos = pos->next)

/**
  * @brief Iterate over a list safe against removal of list entry
  */
#define list_for_each_safe(pos, n, head) \
    for (pos = (head)->next, n = pos->next; pos != (head); \
            pos = n, n = pos->next)

#ifdef __cplusplus
}
#endif  /* __cplusplus */

#endif  /* end of __LIST_H__ */
