/**
  *********************************************************************************
  *
  * @file    .c
  * @brief  Source file
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "usbh_msc_diskio.h"
#include "string.h"
#include "stdio.h"
#include "usb_config.h"
#include "usb_hc.h"
#include "usbh_msc.h"
#include "usb_scsi.h"

/* Private Macros ------------------------------------------------------------ */

#define ES_FATFS_BASE_PATH_MAX_LEN (32)

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS usbh_msc_fatfs_drv_init(BYTE);
DSTATUS usbh_msc_fatfs_drv_stat(BYTE);
DRESULT usbh_msc_fatfs_drv_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT usbh_msc_fatfs_drv_write(BYTE, const BYTE *, DWORD, UINT);
#endif /*_USE_WRITE*/
#if  _USE_IOCTL
    DRESULT usbh_msc_fatfs_drv_ioctl(BYTE, BYTE, void *);
#endif /*_USE_IOCTL*/
/* Private Variables ---------------------------------------------------------*/

static FATFS s_es_fatfs;        /* File system object for NAND flash volume */
static uint8_t s_es_fatfs_path[ES_FATFS_BASE_PATH_MAX_LEN];
static struct usbh_msc *s_es_usb_msc_class1 = NULL;

static const Diskio_drvTypeDef s_usbh_msc_fatfs_drv =
{
    usbh_msc_fatfs_drv_init,
    usbh_msc_fatfs_drv_stat,
    usbh_msc_fatfs_drv_read,

#if  _USE_WRITE
    usbh_msc_fatfs_drv_write,
#endif /*_USE_WRITE*/

#if  _USE_IOCTL
    usbh_msc_fatfs_drv_ioctl,
#endif /*_USE_IOCTL*/
};

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS usbh_msc_fatfs_drv_init(BYTE lun)
{
    if (s_es_usb_msc_class1 == NULL)
        return RES_ERROR;
    else
        return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS usbh_msc_fatfs_drv_stat(BYTE lun)
{
    if (s_es_usb_msc_class1 == NULL)
        return RES_ERROR;
    else
        return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT usbh_msc_fatfs_drv_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    int ret = -1;

    if (s_es_usb_msc_class1)
        ret = usbh_msc_scsi_read10(s_es_usb_msc_class1, sector, buff, count);

    if (ret < 0)
    {
        printf("scsi_read10 error,ret:%d\r\n", ret);
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if _USE_WRITE

DRESULT usbh_msc_fatfs_drv_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    int ret = -1;

    if (s_es_usb_msc_class1)
        ret = usbh_msc_scsi_write10(s_es_usb_msc_class1, sector, buff, count);

    if (ret < 0)
    {
        printf("scsi_write10 error,ret:%d\r\n", ret);
        return RES_ERROR;
    }

    return RES_OK;
}
#endif /*_USE_WRITE*/

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT usbh_msc_fatfs_drv_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    if (s_es_usb_msc_class1 == NULL)
        return RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = s_es_usb_msc_class1->blocknum;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = s_es_usb_msc_class1->blocksize;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = s_es_usb_msc_class1->blocksize;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            res = RES_ERROR;
            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

#endif /*_USE_IOCTL*/

void es_fatfs_error_handler(void)
{
    printf("[es_fatfs_error]");
}

void es_usb_msc_fatfs_register(void)
{
    /* Regeister the disk I/O driver */
    if (FATFS_LinkDriverEx(&s_usbh_msc_fatfs_drv, (char *)s_es_fatfs_path, 0) == 0)
    {
        /* Register the file system object to the FatFs module */
        if (f_mount(&s_es_fatfs, (const TCHAR *)s_es_fatfs_path, 1) != FR_OK)
        {
            /* FatFs Initialization Error */
            es_fatfs_error_handler();

            /*nUnlink the spi Flash disk I/O driver */
            FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
        }
        else
        {
            printf("\r\nfatfs mount succes , path :%s\r\n", s_es_fatfs_path);
        }
    }
}

void es_try_fatfs_unregister(void)
{
    if (f_mount((void *)0, (const TCHAR *)s_es_fatfs_path, 0) != FR_OK)
    {
        /* FatFs Initialization Error */
        es_fatfs_error_handler();
    }

    printf("fatfs unmount succes , path :%s", s_es_fatfs_path);

    /*nUnlink the spi Flash disk I/O driver */
    FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
}

void usbh_msc_run(struct usbh_msc *msc_class)
{
    if (s_es_usb_msc_class1 == NULL)
    {
        s_es_usb_msc_class1 = msc_class;
        es_usb_msc_fatfs_register();
    }
    else
        printf("only support 1 usb_msc_device");
}

void usbh_msc_stop(struct usbh_msc *msc_class)
{
    es_try_fatfs_unregister();
    s_es_usb_msc_class1 = NULL;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
