/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */
#include "main.h"

/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
static uart_handle_t s_h_uart;
static i2c_handle_t s_h_i2c;
/* DMA使用的双缓冲区 4K bytes*/
static uint16_t s_audio_dma_buffer[2 * 1024] __attribute((aligned(4)));

/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */

/**
  * @brief  Redirect fputc to UART
  * @param  ch Character to print
  * @param  f Not used
  * @retval None
  */
int fputc(int ch, FILE *f)
{
    uint8_t x = (uint8_t)ch;
    /* Send one byte */
    ald_uart_send(&s_h_uart, &x, 1, 1000);

    return (ch);
}

uint16_t default_audio_send_data(void *buf, uint16_t length)
{
    printf("feed data\r\n");

    return 0;
}

void init_uart(void)
{
    gpio_init_t gpio_init_x;

    memset(&gpio_init_x, 0, sizeof(gpio_init_x));
    /* Initialize tx pin*/
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_PUSH_PULL;
    gpio_init_x.pupd  = GPIO_PUSH_UP;
    gpio_init_x.podrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.flt   = GPIO_FILTER_DISABLE;
    gpio_init_x.type  = GPIO_TYPE_CMOS;
    gpio_init_x.func  = GPIO_FUNC_3;
    ald_gpio_init(GPIOE, GPIO_PIN_2, &gpio_init_x);

    /* Initialize rx pin*/
    gpio_init_x.mode  = GPIO_MODE_INPUT;
    gpio_init_x.odos  = GPIO_PUSH_PULL;
    gpio_init_x.pupd  = GPIO_PUSH_UP;
    gpio_init_x.func  = GPIO_FUNC_3;
    ald_gpio_init(GPIOE, GPIO_PIN_3, &gpio_init_x);

    /* Initialize UART */
    memset(&s_h_uart, 0x00, sizeof(uart_handle_t));
    /* Initialize uart */
    s_h_uart.perh             = UART2;
    s_h_uart.init.baud        = 115200;
    s_h_uart.init.word_length = UART_WORD_LENGTH_8B;
    s_h_uart.init.stop_bits   = UART_STOP_BITS_1;
    s_h_uart.init.parity      = UART_PARITY_NONE;
    s_h_uart.init.mode        = UART_MODE_UART;
    s_h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    s_h_uart.rx_cplt_cbk = NULL;
    ald_uart_init(&s_h_uart);
}

void audio_i2s_init(uint32_t sample_rate)
{
    struct wm_i2s_info wm_info;
    uint8_t *g_audio_buffer;

    g_audio_buffer = (uint8_t *)s_audio_dma_buffer;
    wm_info.buffers[0] = g_audio_buffer;
    wm_info.buffers[1] = g_audio_buffer + 1024 * 2;
    wm_info.buffer_length = 1024;
    wm_info.spi_x = SPI0;
    wm_info.feed_data = default_audio_send_data;
    wm_info.sample_rate = sample_rate;
    wm_info.dma_tx_ch = 2;
    wm_info.dma_rx_ch = 3;

    if (wm_i2s_init(&wm_info))
    {
        printf("I2S init failed\r\n");
    }
}

/**
  * @brief  初始化I2S0 管脚
  * @retval None
*/
void i2s0_pin_init(void)
{
    gpio_init_t gpio_init_x;

    /* Initialize NSS/WS pin */
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_PUSH_PULL;
    gpio_init_x.pupd  = GPIO_PUSH_UP_DOWN;
    gpio_init_x.podrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.flt   = GPIO_FILTER_DISABLE;
    gpio_init_x.type  = GPIO_TYPE_CMOS;
    gpio_init_x.func  = GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_4, &gpio_init_x);

    /* Initialize SCK/CK pin */
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_PUSH_PULL;
    gpio_init_x.pupd  = GPIO_PUSH_UP_DOWN;
    gpio_init_x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOD, GPIO_PIN_3, &gpio_init_x);

    /* Initialize MISO/MCLK pin */
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_PUSH_PULL;
    gpio_init_x.pupd  = GPIO_PUSH_UP_DOWN;
    gpio_init_x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_4, &gpio_init_x);

    /* Initialize MOSI/SD pin */
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_PUSH_PULL;
    gpio_init_x.pupd  = GPIO_PUSH_UP_DOWN;
    gpio_init_x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_5, &gpio_init_x);

    return;
}

void i2s0_data_in_pin_init(void)
{
    gpio_init_t x;

    /* Initialize MOSI/SD pin */
    x.mode  = GPIO_MODE_INPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_FLOATING;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_5, &x);
}

void i2s0_data_out_pin_init(void)
{
    gpio_init_t gpio_init_x;

    /* Initialize MOSI/SD pin */
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_PUSH_PULL;
    gpio_init_x.pupd  = GPIO_PUSH_UP_DOWN;
    gpio_init_x.podrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.flt   = GPIO_FILTER_DISABLE;
    gpio_init_x.type  = GPIO_TYPE_CMOS;
    gpio_init_x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_5, &gpio_init_x);
}

/**
  * @brief  初始化I2C0
  * @retval None
*/
void i2c0_init()
{
    gpio_init_t gpio_init_x;

    /* Initialize SCK pin */
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_OPEN_DRAIN;
    gpio_init_x.pupd  = GPIO_PUSH_UP;
    gpio_init_x.podrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.flt   = GPIO_FILTER_DISABLE;
    gpio_init_x.type  = GPIO_TYPE_CMOS;
    gpio_init_x.func  = GPIO_FUNC_5;
    ald_gpio_init(GPIOB, GPIO_PIN_6, &gpio_init_x);

    /* Initialize SDA pin */
    gpio_init_x.mode  = GPIO_MODE_OUTPUT;
    gpio_init_x.odos  = GPIO_OPEN_DRAIN;
    gpio_init_x.pupd  = GPIO_PUSH_UP;
    gpio_init_x.func  = GPIO_FUNC_5;
    ald_gpio_init(GPIOB, GPIO_PIN_7, &gpio_init_x);

    memset(&s_h_i2c, 0, sizeof(s_h_i2c));
    s_h_i2c.perh = I2C0;
    s_h_i2c.init.module  = I2C_MODULE_MASTER;
    s_h_i2c.init.addr_mode    = I2C_ADDR_7BIT;
    s_h_i2c.init.clk_speed    = 400000;
    s_h_i2c.init.dual_addr    = I2C_DUALADDR_ENABLE;
    s_h_i2c.init.general_call = I2C_GENERALCALL_ENABLE;
    s_h_i2c.init.no_stretch   = I2C_NOSTRETCH_DISABLE;
    s_h_i2c.init.own_addr1    = 0xA0;
    ald_i2c_init(&s_h_i2c);
}

void usb_pin_init(void)
{
    gpio_init_t gpio_init_x;

    /* Initialize vbus pin */
    gpio_init_x.mode = GPIO_MODE_OUTPUT;
    gpio_init_x.odos = GPIO_PUSH_PULL;
    gpio_init_x.pupd = GPIO_PUSH_UP;
    gpio_init_x.podrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init_x.flt = GPIO_FILTER_DISABLE;
    gpio_init_x.type = GPIO_TYPE_TTL;
    gpio_init_x.func = GPIO_FUNC_5;
    ald_gpio_init(GPIOB, GPIO_PIN_15, &gpio_init_x);

    return;
}

void usb_dc_low_level_init(void)
{
    ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
    ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
    ald_rmu_reset_periperal(RMU_PERH_USB);
    ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
    ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
    usb_pin_init();

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    ald_delay_ms(1000);

    init_uart();
    printf("Startup...\r\n");
    ald_delay_ms(100);

    /* 初始化WM8978 */
    i2c0_init();

    if (WM8978_Init(&s_h_i2c))
        printf("Error WM8978 init\r\n");

    /* 初始化I2S */
    i2s0_pin_init();

    /* 音频初始化信息 */
    audio_i2s_init(16000);

    printf("Audio init ok\r\n");
    ald_delay_ms(100);

    es_usbd_audio_init();

    while (1)
    {
        es_usbd_audio_loop();
    }
}

/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
