#include "usbd_core.h"
#include "usbd_audio.h"
#include "main.h"

#define USBD_VID           0x30cc
#define USBD_PID           0xae06

#define USBD_MAX_POWER     100
#define USBD_LANGID_STRING 1033

#ifdef CONFIG_USB_HS
    #define EP_INTERVAL 0x04
#else
    #define EP_INTERVAL 0x01
#endif

#define AUDIO_IN_EP  0x81
#define AUDIO_OUT_EP 0x02


#define AUDIO_IN_FU_ID  0x02
#define AUDIO_OUT_FU_ID 0x05

/* AUDIO Class Config */
#define AUDIO_SPEAKER_FREQ            16000U
#define AUDIO_SPEAKER_FRAME_SIZE_BYTE 2U
#define AUDIO_SPEAKER_RESOLUTION_BIT  16U
#define AUDIO_MIC_FREQ                16000U
#define AUDIO_MIC_FRAME_SIZE_BYTE     2U
#define AUDIO_MIC_RESOLUTION_BIT      16U

#define AUDIO_SAMPLE_FREQ(frq) (uint8_t)(frq), (uint8_t)((frq >> 8)), (uint8_t)((frq >> 16))

/* AudioFreq * DataSize (2 bytes) * NumChannels (Stereo: 2) */
#define AUDIO_OUT_PACKET ((uint32_t)((AUDIO_SPEAKER_FREQ * AUDIO_SPEAKER_FRAME_SIZE_BYTE * 2) / 1000))
/* 16bit(2 Bytes) 双声道(Mono:2) */
#define AUDIO_IN_PACKET ((uint32_t)((AUDIO_MIC_FREQ * AUDIO_MIC_FRAME_SIZE_BYTE * 2) / 1000))

#define ES_USB_AUDIO_STREAM_BUF_SIZE (8192)
#define ES_USB_AUDIO_PLAY_BUF_SIZE (AUDIO_OUT_PACKET)

#define USB_AUDIO_CONFIG_DESC_SIZ (unsigned long)(9 +                                       \
        AUDIO_AC_DESCRIPTOR_INIT_LEN(2) +         \
        AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
        AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
        AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC +    \
        AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
        AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
        AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC +    \
        AUDIO_AS_DESCRIPTOR_INIT_LEN(1) +         \
        AUDIO_AS_DESCRIPTOR_INIT_LEN(1))

#define AUDIO_AC_SIZ (AUDIO_SIZEOF_AC_HEADER_DESC(2) +          \
                      AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
                      AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
                      AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC +    \
                      AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
                      AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
                      AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC)

const uint8_t audio_descriptor[] =
{
    USB_DEVICE_DESCRIPTOR_INIT(USB_2_0, 0xef, 0x02, 0x01, USBD_VID, USBD_PID, 0x0001, 0x01),
    USB_CONFIG_DESCRIPTOR_INIT(USB_AUDIO_CONFIG_DESC_SIZ, 0x03, 0x01, USB_CONFIG_BUS_POWERED, USBD_MAX_POWER),
    AUDIO_AC_DESCRIPTOR_INIT(0x00, 0x03, AUDIO_AC_SIZ, 0x00, 0x01, 0x02),
    AUDIO_AC_INPUT_TERMINAL_DESCRIPTOR_INIT(0x01, AUDIO_INTERM_MIC, 0x02, 0x0003),
    AUDIO_AC_FEATURE_UNIT_DESCRIPTOR_INIT(0x02, 0x01, 0x01, 0x03, 0x00, 0x00),
    AUDIO_AC_OUTPUT_TERMINAL_DESCRIPTOR_INIT(0x03, AUDIO_TERMINAL_STREAMING, 0x02),
    AUDIO_AC_INPUT_TERMINAL_DESCRIPTOR_INIT(0x04, AUDIO_TERMINAL_STREAMING, 0x02, 0x0003),
    AUDIO_AC_FEATURE_UNIT_DESCRIPTOR_INIT(0x05, 0x04, 0x01, 0x03, 0x00, 0x00),
    AUDIO_AC_OUTPUT_TERMINAL_DESCRIPTOR_INIT(0x06, AUDIO_OUTTERM_SPEAKER, 0x05),
    AUDIO_AS_DESCRIPTOR_INIT(0x01, 0x04, 0x02, AUDIO_SPEAKER_FRAME_SIZE_BYTE, AUDIO_SPEAKER_RESOLUTION_BIT, AUDIO_OUT_EP, 0x09, AUDIO_OUT_PACKET,
                             EP_INTERVAL, AUDIO_SAMPLE_FREQ_3B(AUDIO_SPEAKER_FREQ)),
    AUDIO_AS_DESCRIPTOR_INIT(0x02, 0x03, 0x02, AUDIO_MIC_FRAME_SIZE_BYTE, AUDIO_MIC_RESOLUTION_BIT, AUDIO_IN_EP, 0x05, AUDIO_IN_PACKET,
                             EP_INTERVAL, AUDIO_SAMPLE_FREQ_3B(AUDIO_MIC_FREQ)),
    ///////////////////////////////////////
    /// string0 descriptor
    ///////////////////////////////////////
    USB_LANGID_INIT(USBD_LANGID_STRING),
    ///////////////////////////////////////
    /// string1 descriptor
    ///////////////////////////////////////
    0xE,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    'e', 0x00,                  /* wcChar0 */
    's', 0x00,                  /* wcChar1 */
    's', 0x00,                  /* wcChar2 */
    'e', 0x00,                  /* wcChar3 */
    'm', 0x00,                  /* wcChar4 */
    'i', 0x00,                  /* wcChar5 */
    ///////////////////////////////////////
    /// string2 descriptor
    ///////////////////////////////////////
    0x20,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    'e', 0x00,                  /* wcChar0 */
    's', 0x00,                  /* wcChar1 */
    's', 0x00,                  /* wcChar2 */
    'e', 0x00,                  /* wcChar3 */
    'm', 0x00,                  /* wcChar4 */
    'i', 0x00,                  /* wcChar5 */
    ' ', 0x00,                  /* wcChar9 */
    'U', 0x00,                  /* wcChar10 */
    'A', 0x00,                  /* wcChar11 */
    'C', 0x00,                  /* wcChar12 */
    ' ', 0x00,                  /* wcChar13 */
    'D', 0x00,                  /* wcChar14 */
    'E', 0x00,                  /* wcChar15 */
    'M', 0x00,                  /* wcChar16 */
    'O', 0x00,                  /* wcChar17 */
    ///////////////////////////////////////
    /// string3 descriptor
    ///////////////////////////////////////
    0x16,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    '2', 0x00,                  /* wcChar0 */
    '0', 0x00,                  /* wcChar1 */
    '2', 0x00,                  /* wcChar2 */
    '2', 0x00,                  /* wcChar3 */
    '1', 0x00,                  /* wcChar4 */
    '2', 0x00,                  /* wcChar5 */
    '1', 0x00,                  /* wcChar6 */
    '2', 0x00,                  /* wcChar7 */
    '0', 0x00,                  /* wcChar8 */
    '1', 0x00,                  /* wcChar9 */
#ifdef CONFIG_USB_HS
    ///////////////////////////////////////
    /// device qualifier descriptor
    ///////////////////////////////////////
    0x0a,
    USB_DESCRIPTOR_TYPE_DEVICE_QUALIFIER,
    0x00,
    0x02,
    0xEF,
    0x02,
    0x01,
    0x40,
    0x01,
    0x00,
#endif
    0x00
};

extern int usbd_read_packet(uint8_t ep_addr, uint8_t *buffer, uint16_t len);
extern int usbd_write_packet(uint8_t ep_addr, uint8_t *buffer, uint16_t len);

#define ES_AUDIO_DATA_BUF_SIZE (4096)
#if(ES_AUDIO_DATA_BUF_SIZE == (4096))
    #define ES_AUDIO_DATA_BUF_SIZE_GET_INDEX(NUM_INDEX)  ((NUM_INDEX)&0xFFF)
#else
    #define ES_AUDIO_DATA_BUF_SIZE_GET_INDEX(NUM_INDEX)  ((NUM_INDEX)%(ES_AUDIO_DATA_BUF_SIZE))
#endif

static uint8_t s_es_usbd_audio_play_flag = 0;
static uint8_t s_es_usbd_audio_record_flag = 0;
static uint32_t s_es_true_audio_record_buffer[(ES_AUDIO_DATA_BUF_SIZE + 0x3) >> 2];
static uint8_t *s_es_audio_record_buffer = (uint8_t *)(&(s_es_true_audio_record_buffer[0]));
static uint32_t s_es_audio_record_rx_index = 0U;
static uint32_t s_es_audio_record_tx_index = 0U;
static uint8_t s_es_audio_out_vol = 0U;
static uint8_t s_es_audio_in_vol = 0U;
static uint8_t s_es_usbd_configure_done_flag = 1U;

static struct usbd_interface s_audio_intf1, s_audio_intf2, s_audio_intf3;

static struct usbd_endpoint s_audio_in_ep =
{
    .ep_cb = NULL,
    .ep_addr = AUDIO_IN_EP
};

static struct usbd_endpoint s_audio_out_ep =
{
    .ep_cb = NULL,
    .ep_addr = AUDIO_OUT_EP
};

static uint32_t es_get_tx_buf_para(uint32_t *tx_num, uint32_t *rx_num, uint32_t buf_size)
{
    uint32_t rx_index_in_buf, tx_index_in_buf;

    rx_index_in_buf = (*rx_num) % buf_size;
    tx_index_in_buf = (*tx_num) % buf_size;

    if (rx_index_in_buf < tx_index_in_buf)
    {
        return (buf_size - tx_index_in_buf);
    }
    else if (rx_index_in_buf > tx_index_in_buf)
    {
        return (rx_index_in_buf - tx_index_in_buf);
    }
    else
    {
        *tx_num = *rx_num;
        return 0U;
    }

}

uint16_t audio_play_data(void *buf, uint16_t length)
{
    int ret;
    uint32_t i;
    uint16_t *buf16 = buf;
    uint8_t *buf8 = buf;
    static uint32_t last_voice = 0U;

    if (((uint32_t)buf) & 0x3)
        printf("buf not align 4\r\n");

    if(s_es_usbd_audio_play_flag)
    {
        ret = usbd_read_packet(AUDIO_OUT_EP, buf8, 512);

        if (ret > 512) /*写数组溢出*/
        {
            while (1);
        }

        if (ret > 0)
        {
            last_voice = *(uint32_t *)(((uint32_t)(buf8 + ret - 1)) & 0xFFFFFFFC);

            return (uint16_t)(ret / 2);
        }
        else
        {
            /*无数据可播放时：插入一段最近的声音*/
            for (i = 0; i < 64; i += 4)
                * (uint32_t *)((uint32_t)(buf) + i) = last_voice;  
            
            last_voice = 0U;

            return 32;
        }
    }

    return 0;
}

uint16_t audio_record_data(void *buf, uint16_t length)
{
    uint32_t i;
    uint16_t *buf16 = (uint16_t *)buf;

    for (i = 0; i < length; i++)
    {
        *(uint16_t *)((uint32_t)s_es_audio_record_buffer + ES_AUDIO_DATA_BUF_SIZE_GET_INDEX(s_es_audio_record_rx_index)) = buf16[i];
        s_es_audio_record_rx_index += 2;
    }

    return length;
}

void usbd_audio_open(uint8_t intf)
{
    if (intf == 1)
    {
        if (s_es_usbd_audio_play_flag == 0)
        {
            if (wm_status())
                wm_stop();

            i2s0_data_out_pin_init();
            wm_set_mode(WM_SEND);
            wm_set_callback(audio_play_data);
            wm_play();
        }

        s_es_usbd_audio_play_flag = 1;
    }
    else
    {
        if (s_es_usbd_audio_record_flag == 0)
        {
            if (wm_status())
                wm_stop();

            s_es_audio_in_vol = 0x2F;
            wm_set_vol(0x4, s_es_audio_in_vol);

            i2s0_data_in_pin_init();
            wm_set_mode(WM_RECV);
            wm_set_callback(audio_record_data);
            wm_record();
        }

        s_es_usbd_audio_record_flag = 1;
    }

    printf("OPEN %d\r\n", intf);
}

void usbd_audio_close(uint8_t intf)
{
    wm_stop();

    if (intf == 1)
    {
        s_es_audio_out_vol = 0;
        wm_set_vol(0x3, s_es_audio_out_vol);
        s_es_usbd_audio_play_flag = 0;
    }
    else
    {
        s_es_audio_in_vol = 0;
        wm_set_vol(0x4, s_es_audio_in_vol);
        s_es_usbd_audio_record_flag = 0;
    }

    printf("CLOSE %d\r\n", intf);
}

void usbd_audio_set_volume(uint8_t entity_id, uint8_t ch, int volume)
{
    /*  0x000000 = -57dB , 0x111001 = 0dB , 0x111111 = +6dB  */
    s_es_audio_out_vol = ((uint8_t)(volume + 57.0)) & 0x3F;
    wm_set_vol(0x3, s_es_audio_out_vol);

    /*  0x000000=-12dB, 0x000001=-11.25dB , 0x010000=0dB , 0x111111=35.25dB  */
    wm_set_vol(0x4, s_es_audio_in_vol);
    printf("set_volume:entity_id = %d,ch = %d,volume = %d\r\n", entity_id, ch, volume);
}

void usbd_audio_set_mute(uint8_t entity_id, uint8_t ch, bool mute)
{
    if (mute)
        wm_set_vol(0x7, 0);
    else
    {
        wm_set_vol(0x3, s_es_audio_out_vol);
        wm_set_vol(0x4, s_es_audio_in_vol);
    }

    printf("set_mute:entity_id = %d,ch = %d,mute = %d\r\n", entity_id, ch, mute);
}

void usbd_audio_set_sampling_freq(uint8_t ep, uint32_t sampling_freq)
{
    printf("set_sampling_freq:ep = %d,sampling_freq = %d\r\n", ep, (unsigned int)sampling_freq);
}

void usbd_audio_set_pitch(uint8_t ep, bool enable)
{
    printf("set_pitch:ep = %d,enable = %d\r\n", ep, enable);
}

void es_usbd_audio_loop(void)
{
    int ret;
    uint32_t len, last_rx_index;

    last_rx_index = s_es_audio_record_rx_index;

    if (last_rx_index != s_es_audio_record_tx_index)
    {
        len = es_get_tx_buf_para(&s_es_audio_record_tx_index, &last_rx_index, ES_AUDIO_DATA_BUF_SIZE);
        ret = usbd_write_packet(AUDIO_IN_EP, &(s_es_audio_record_buffer[(s_es_audio_record_tx_index) % (ES_AUDIO_DATA_BUF_SIZE)]), len);

        if (ret > 0)
        {
            s_es_audio_record_tx_index += ret;
        }
    }
}

void usbd_event_handler(uint8_t event)
{
    switch (event) 
	{
        case USBD_EVENT_RESET:
            break;
        case USBD_EVENT_CONNECTED:
            break;
        case USBD_EVENT_DISCONNECTED:
            break;
        case USBD_EVENT_RESUME:
            break;
        case USBD_EVENT_SUSPEND:
            break;
        case USBD_EVENT_CONFIGURED:
		{
			s_es_usbd_configure_done_flag = 0;
			
            break;
		}
        case USBD_EVENT_SET_REMOTE_WAKEUP:
            break;
        case USBD_EVENT_CLR_REMOTE_WAKEUP:
            break;

        default:
            break;
    }
}

struct audio_entity_info audio_entity_table[] = {
    { .bEntityId = AUDIO_IN_FU_ID,
      .bDescriptorSubtype = AUDIO_CONTROL_FEATURE_UNIT,
      .ep = AUDIO_IN_EP },
    { .bEntityId = AUDIO_OUT_FU_ID,
      .bDescriptorSubtype = AUDIO_CONTROL_FEATURE_UNIT,
      .ep = AUDIO_OUT_EP },
};

int es_usbd_audio_init(void)
{
    usbd_desc_register(audio_descriptor);
    usbd_add_interface(usbd_audio_init_intf(&s_audio_intf1, 0x0100, audio_entity_table, 2));
    usbd_add_interface(usbd_audio_init_intf(&s_audio_intf2, 0x0100, audio_entity_table, 2));
    usbd_add_interface(usbd_audio_init_intf(&s_audio_intf3, 0x0100, audio_entity_table, 2));
    usbd_add_endpoint(&s_audio_in_ep);
    usbd_add_endpoint(&s_audio_out_ep);

    usbd_initialize();

    while (s_es_usbd_configure_done_flag) {}

    return 0;
}

__WEAK void vPortEnterCritical(void)
{

}

__WEAK void vPortExitCritical(void)
{

}

__WEAK void vTaskSuspendAll(void)
{

}

__WEAK void xTaskResumeAll(void)
{

}

