/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          26 Oct 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "winusb_custom.h"
#include "usbd_core.h"
#include "usbd_cdc.h"

/* Private Macros ------------------------------------------------------------ */
#define WCID_VENDOR_CODE 0x01

/* Private function prototypes ----------------------------------------------- */
static void usbd_winusb_out(uint8_t ep, uint32_t nbytes);
static void usbd_winusb_in(uint8_t ep, uint32_t nbytes);

/* Private Variables --------------------------------------------------------- */
static __ALIGN_BEGIN const uint8_t WCID_StringDescriptor_MSOS[18] __ALIGN_END = {
    ///////////////////////////////////////
    /// MS OS string descriptor
    ///////////////////////////////////////
    0x12,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    /* MSFT100 */
    'M', 0x00, 'S', 0x00, 'F', 0x00, 'T', 0x00, /* wcChar_7 */
    '1', 0x00, '0', 0x00, '0', 0x00,            /* wcChar_7 */
    WCID_VENDOR_CODE,                           /* bVendorCode */
    0x00,                                       /* bReserved */
};

static __ALIGN_BEGIN const uint8_t WINUSB_WCIDDescriptor[40] __ALIGN_END = {
    ///////////////////////////////////////
    /// WCID descriptor
    ///////////////////////////////////////
    0x28, 0x00, 0x00, 0x00,                   /* dwLength */
    0x00, 0x01,                               /* bcdVersion */
    0x04, 0x00,                               /* wIndex */
    0x01,                                     /* bCount */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* bReserved_7 */

    ///////////////////////////////////////
    /// WCID function descriptor
    ///////////////////////////////////////
    0x02, /* bFirstInterfaceNumber */
    0x01, /* bReserved */
    /* WINUSB */
    'W', 'I', 'N', 'U', 'S', 'B', 0x00, 0x00, /* cCID_8 */
    /*  */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* cSubCID_8 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,             /* bReserved_6 */
};

static __ALIGN_BEGIN const uint8_t WINUSB_IF0_WCIDProperties[142] __ALIGN_END = {
    ///////////////////////////////////////
    /// WCID property descriptor
    ///////////////////////////////////////
    0x8e, 0x00, 0x00, 0x00, /* dwLength */
    0x00, 0x01,             /* bcdVersion */
    0x05, 0x00,             /* wIndex */
    0x01, 0x00,             /* wCount */

    ///////////////////////////////////////
    /// registry propter descriptor
    ///////////////////////////////////////
    0x84, 0x00, 0x00, 0x00, /* dwSize */
    0x01, 0x00, 0x00, 0x00, /* dwPropertyDataType */
    0x28, 0x00,             /* wPropertyNameLength */
    /* DeviceInterfaceGUID */
    'D', 0x00, 'e', 0x00, 'v', 0x00, 'i', 0x00,  /* wcName_20 */
    'c', 0x00, 'e', 0x00, 'I', 0x00, 'n', 0x00,  /* wcName_20 */
    't', 0x00, 'e', 0x00, 'r', 0x00, 'f', 0x00,  /* wcName_20 */
    'a', 0x00, 'c', 0x00, 'e', 0x00, 'G', 0x00,  /* wcName_20 */
    'U', 0x00, 'I', 0x00, 'D', 0x00, 0x00, 0x00, /* wcName_20 */
    0x4e, 0x00, 0x00, 0x00,                      /* dwPropertyDataLength */
    /* {CDB3B5AD-293B-4663-AA36-1AAE46463776} */
    '{', 0x00, 'C', 0x00, 'D', 0x00, 'B', 0x00, /* wcData_39 */
    '3', 0x00, 'B', 0x00, '5', 0x00, 'A', 0x00, /* wcData_39 */
    'D', 0x00, '-', 0x00, '2', 0x00, '9', 0x00, /* wcData_39 */
    '3', 0x00, 'B', 0x00, '-', 0x00, '4', 0x00, /* wcData_39 */
    '6', 0x00, '6', 0x00, '3', 0x00, '-', 0x00, /* wcData_39 */
    'A', 0x00, 'A', 0x00, '3', 0x00, '6', 0x00, /* wcData_39 */
    '-', 0x00, '1', 0x00, 'A', 0x00, 'A', 0x00, /* wcData_39 */
    'E', 0x00, '4', 0x00, '6', 0x00, '4', 0x00, /* wcData_39 */
    '6', 0x00, '3', 0x00, '7', 0x00, '7', 0x00, /* wcData_39 */
    '6', 0x00, '}', 0x00, 0x00, 0x00,           /* wcData_39 */
};

static uint8_t s_winusb_buffer[1024] = {0};      
static uint8_t s_winusb_buffer_temp[1024] = {0}; 
 
static struct ringbuffer s_winusb_ringbuffer[1];

/* Public Variables ---------------------------------------------------------- */
const uint8_t * WINUSB_IFx_WCIDProperties[] = {
    0,                  
    0,
    WINUSB_IF0_WCIDProperties,     /*WINUSB_WCIDDescriptor: bFirstInterfaceNumber[1] = 0x2 */
};

struct usb_msosv1_descriptor g_msosv1_desc =
{
    .string = WCID_StringDescriptor_MSOS,
    .vendor_code = WCID_VENDOR_CODE,
    .compat_id = WINUSB_WCIDDescriptor,
    .comp_id_property = WINUSB_IFx_WCIDProperties,
};

struct usbd_endpoint g_winusb_out_ep = {
    .ep_addr = USBD_IF2_AL0_EP0_ADDR,
    .ep_cb = usbd_winusb_out
};

struct usbd_endpoint g_winusb_in_ep = {
    .ep_addr = USBD_IF2_AL0_EP1_ADDR,
    .ep_cb = usbd_winusb_in
};


/* Private Constants --------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
static void usbd_winusb_out(uint8_t ep, uint32_t nbytes)
{
    return;
}

static void usbd_winusb_in(uint8_t ep, uint32_t nbytes)
{
    return;
}

/**
  * @brief  Initializate ringbuffer for cdc_acm.
  * @retval None
  */
void winusb_ringbuffer_init(void)
{
    ringbuffer_init(s_winusb_ringbuffer, s_winusb_buffer, sizeof(s_winusb_buffer));
    
    return;
}

/**
  * @brief            receive packet and return to USB host
  * @pre              none
  * @param[in]        none
  * @retval           none
  */
void winusb_schedual(void)
{
    uint16_t len = ringbuffer_space_len(s_winusb_ringbuffer);
    if (len >= usb_ep_out_data_avail(cmsis_dap_handle.winusb_ep[0].ep_addr) && len)
    {
        len = usb_ep_out_data_avail(cmsis_dap_handle.winusb_ep[0].ep_addr);
        
        usbd_read_packet(cmsis_dap_handle.winusb_ep[0].ep_addr, s_winusb_buffer_temp, len);
        ringbuffer_put(s_winusb_ringbuffer, s_winusb_buffer_temp, len);
    }
    
    len = ringbuffer_data_len(s_winusb_ringbuffer);
    if (len)
    {
        len = len > usb_ep_in_data_avail(cmsis_dap_handle.winusb_ep[0].ep_addr) ? usb_ep_in_data_avail(cmsis_dap_handle.hid_1_ep[0].ep_addr) : len; 
        
        ringbuffer_get(s_winusb_ringbuffer, s_winusb_buffer_temp, len);
        usbd_write_packet(cmsis_dap_handle.winusb_ep[1].ep_addr, s_winusb_buffer_temp, len);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
