/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 */

#include "ringbuffer.h"
#include "ald_conf.h"

static inline enum_ringbuffer_state ringbuffer_status(struct ringbuffer *rb)
{
    uint32_t mask = __get_PRIMASK();
    __set_PRIMASK(1);
    if (rb->read_index == rb->write_index)
    {
        if (rb->read_mirror == rb->write_mirror){
            __set_PRIMASK(mask);
            return RINGBUFFER_EMPTY;
        }
        else{
            __set_PRIMASK(mask);
            return RINGBUFFER_FULL;
        }
    }
    __set_PRIMASK(mask);
    return RINGBUFFER_HALFFULL;
}

unsigned long ringbuffer_init(struct ringbuffer *rb,
                        uint8_t           *pool,
                        int16_t            size)
{
    ringbuffer_assert_param(rb != NULL);
    ringbuffer_assert_param(size > 0);

    /* initialize read and write index */
    rb->read_mirror = rb->read_index = 0;
    rb->write_mirror = rb->write_index = 0;

    /* set buffer pool and size, size is alligned as 4 bytes */
    rb->buffer_ptr = pool;
    rb->buffer_size = ((size) & ~((4) - 1));
	
	return rb->buffer_size;
}

/**
 * put a block of data into ring buffer
 */
unsigned long ringbuffer_put(struct ringbuffer *rb,
                            const uint8_t     *ptr,
                            uint16_t           length)
{
    uint16_t size;
    uint32_t mask = __get_PRIMASK();
    
    __set_PRIMASK(1);

    ringbuffer_assert_param(rb != NULL);

    /* whether has enough space */
    size = ringbuffer_space_len(rb);

    /* no space */
    if (size == 0){
        __set_PRIMASK(mask);
        return 0;
    }

    /* drop some data */
    if (size < length)
        length = size;

    if (rb->buffer_size - rb->write_index > length)
    {
        /* read_index - write_index = empty space */
        memcpy(&rb->buffer_ptr[rb->write_index], ptr, length);
        /* this should not cause overflow because there is enough space for
         * length of data in current mirror */
        rb->write_index += length; 
        
        __set_PRIMASK(mask);
        return length;
    }

    memcpy(&rb->buffer_ptr[rb->write_index],
           &ptr[0],
           rb->buffer_size - rb->write_index);
    memcpy(&rb->buffer_ptr[0],
           &ptr[rb->buffer_size - rb->write_index],
           length - (rb->buffer_size - rb->write_index));

    /* we are going into the other side of the mirror */
    rb->write_mirror = ~rb->write_mirror;
    rb->write_index = length - (rb->buffer_size - rb->write_index);
    
    __set_PRIMASK(mask);
    return length;
}

/**
 * put a block of data into ring buffer
 *
 * When the buffer is full, it will discard the old data.
 */
unsigned long ringbuffer_put_force(struct ringbuffer *rb,
                            const uint8_t     *ptr,
                            uint16_t           length)
{
    uint16_t space_length;
    uint32_t mask = __get_PRIMASK();

    __set_PRIMASK(1);
    
    ringbuffer_assert_param(rb != NULL);

    space_length = ringbuffer_space_len(rb);

    if (length > rb->buffer_size)
    {
        ptr = &ptr[length - rb->buffer_size];
        length = rb->buffer_size;
    }

    if (rb->buffer_size - rb->write_index > length)
    {
        /* read_index - write_index = empty space */
        memcpy(&rb->buffer_ptr[rb->write_index], ptr, length);
        /* this should not cause overflow because there is enough space for
         * length of data in current mirror */
        rb->write_index += length;

        if (length > space_length)
            rb->read_index = rb->write_index;
       
        __set_PRIMASK(mask);
        return length;
    }

    memcpy(&rb->buffer_ptr[rb->write_index],
           &ptr[0],
           rb->buffer_size - rb->write_index);
    memcpy(&rb->buffer_ptr[0],
           &ptr[rb->buffer_size - rb->write_index],
           length - (rb->buffer_size - rb->write_index));

    /* we are going into the other side of the mirror */
    rb->write_mirror = ~rb->write_mirror;
    rb->write_index = length - (rb->buffer_size - rb->write_index);

    if (length > space_length)
    {
        if (rb->write_index <= rb->read_index)
            rb->read_mirror = ~rb->read_mirror;
        rb->read_index = rb->write_index;
    }
    
    __set_PRIMASK(mask);
    return length;
}


/**
 *  get data from ring buffer
 */
unsigned long ringbuffer_get(struct ringbuffer *rb,
                            uint8_t           *ptr,
                            uint16_t           length)
{
    unsigned long size;
    uint32_t mask = __get_PRIMASK();

    __set_PRIMASK(1);
    
    ringbuffer_assert_param(rb != NULL);

    /* whether has enough data  */
    size = ringbuffer_data_len(rb);

    /* no data */
    if (size == 0){
        __set_PRIMASK(mask);
        return 0;
    }

    /* less data */
    if (size < length)
        length = size;

    if (rb->buffer_size - rb->read_index > length)
    {
        /* copy all of data */
        memcpy(ptr, &rb->buffer_ptr[rb->read_index], length);
        /* this should not cause overflow because there is enough space for
         * length of data in current mirror */
        rb->read_index += length;
        __set_PRIMASK(mask);
        return length;
    }

    memcpy(&ptr[0],
           &rb->buffer_ptr[rb->read_index],
           rb->buffer_size - rb->read_index);
    memcpy(&ptr[rb->buffer_size - rb->read_index],
           &rb->buffer_ptr[0],
           length - (rb->buffer_size - rb->read_index));

    /* we are going into the other side of the mirror */
    rb->read_mirror = ~rb->read_mirror;
    rb->read_index = length - (rb->buffer_size - rb->read_index);
    
    __set_PRIMASK(mask);
    return length;
}


/**
 *  peak data from ring buffer
 */
unsigned long ringbuffer_peak(struct ringbuffer *rb, uint8_t **ptr)
{
    uint32_t mask = __get_PRIMASK();
    ringbuffer_assert_param(rb != NULL);

    *ptr = NULL;

    /* whether has enough data  */
    unsigned long size = ringbuffer_data_len(rb);

    __set_PRIMASK(1);
    
    /* no data */
    if (size == 0)
        return 0;

    *ptr = &rb->buffer_ptr[rb->read_index];

    if(rb->buffer_size - rb->read_index > size)
    {
        rb->read_index += size;
        __set_PRIMASK(mask);
        return size;
    }

    size = rb->buffer_size - rb->read_index;

    /* we are going into the other side of the mirror */
    rb->read_mirror = ~rb->read_mirror;
    rb->read_index = 0;
    
    __set_PRIMASK(mask);
    return size;
}


/**
 * put a character into ring buffer
 */
unsigned long ringbuffer_putchar(struct ringbuffer *rb, const uint8_t ch)
{
    uint32_t mask = __get_PRIMASK();
    ringbuffer_assert_param(rb != NULL);

    __set_PRIMASK(1);
    
    /* whether has enough space */
    if (!ringbuffer_space_len(rb)){
        __set_PRIMASK(mask);
        return 0;
    }

    rb->buffer_ptr[rb->write_index] = ch;

    /* flip mirror */
    if (rb->write_index == rb->buffer_size-1)
    {
        rb->write_mirror = ~rb->write_mirror;
        rb->write_index = 0;
    }
    else
    {
        rb->write_index++;
    }
    
    __set_PRIMASK(mask);
    return 1;
}


/**
 * put a character into ring buffer
 *
 * When the buffer is full, it will discard one old data.
 */
unsigned long ringbuffer_putchar_force(struct ringbuffer *rb, const uint8_t ch)
{
    enum_ringbuffer_state old_state;
    uint32_t mask = __get_PRIMASK();

    __set_PRIMASK(1);
    
    ringbuffer_assert_param(rb != NULL);

    old_state = ringbuffer_status(rb);

    rb->buffer_ptr[rb->write_index] = ch;

    /* flip mirror */
    if (rb->write_index == rb->buffer_size-1)
    {
        rb->write_mirror = ~rb->write_mirror;
        rb->write_index = 0;
        if (old_state == RINGBUFFER_FULL)
        {
            rb->read_mirror = ~rb->read_mirror;
            rb->read_index = rb->write_index;
        }
    }
    else
    {
        rb->write_index++;
        if (old_state == RINGBUFFER_FULL)
            rb->read_index = rb->write_index;
    }
    
    __set_PRIMASK(mask);
    return 1;
}


/**
 * get a character from a ringbuffer
 */
unsigned long ringbuffer_getchar(struct ringbuffer *rb, uint8_t *ch)
{
    uint32_t mask = __get_PRIMASK();
    ringbuffer_assert_param(rb != NULL);

    __set_PRIMASK(1);
    
    /* ringbuffer is empty */
    if (!ringbuffer_data_len(rb)){
        __set_PRIMASK(mask);
        return 0;
    }

    /* put character */
    *ch = rb->buffer_ptr[rb->read_index];

    if (rb->read_index == rb->buffer_size-1)
    {
        rb->read_mirror = ~rb->read_mirror;
        rb->read_index = 0;
    }
    else
    {
        rb->read_index++;
    }
    
    __set_PRIMASK(mask);
    return 1;
}


/**
 * get the size of data in rb
 */
unsigned long ringbuffer_data_len(struct ringbuffer *rb)
{
    uint32_t mask = __get_PRIMASK();
    __set_PRIMASK(1);
    
    switch (ringbuffer_status(rb))
    {
    case RINGBUFFER_EMPTY:
        __set_PRIMASK(mask);
        return 0;
    case RINGBUFFER_FULL:
        __set_PRIMASK(mask);
        return rb->buffer_size;
    case RINGBUFFER_HALFFULL:
    default:
    {
        unsigned long wi = rb->write_index, ri = rb->read_index;
        
        __set_PRIMASK(mask);

        if (wi > ri)
            return wi - ri;
        else
            return rb->buffer_size - (ri - wi);
    }
    }
}


/**
 * empty the rb
 */
void ringbuffer_reset(struct ringbuffer *rb)
{
    ringbuffer_assert_param(rb != NULL);

    rb->read_mirror = 0;
    rb->read_index = 0;
    rb->write_mirror = 0;
    rb->write_index = 0;
	
}


#ifdef RINGBUFFER_USING_HEAP

struct ringbuffer* ringbuffer_create(uint16_t size)
{
    struct ringbuffer *rb;
    uint8_t *pool;

    assert_param(size > 0);

    size = RT_ALIGN_DOWN(size, RT_ALIGN_SIZE);

    rb = (struct ringbuffer *)malloc(sizeof(struct ringbuffer));
    if (rb == NULL)
        goto exit;

    pool = (uint8_t *)malloc(size);
    if (pool == NULL)
    {
        free(rb);
        rb = NULL;
        goto exit;
    }
    ringbuffer_init(rb, pool, size);

exit:
    return rb;
}


void ringbuffer_destroy(struct ringbuffer *rb)
{
    assert_param(rb != NULL);

    free(rb->buffer_ptr);
    free(rb);
}


#endif
