/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          26 Oct 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "usbd_core.h"
#include "usbd_hid.h"
#include "hid_custom.h"
#include "cmsis_dap_demo.h"

/* Private Macros ------------------------------------------------------------ */
/*!< hid state ! Data can be sent only when state is idle  */ 
#define HID_STATE_IDLE 0
#define HID_STATE_BUSY 1


/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
static volatile uint8_t s_custom_state_1 = HID_STATE_IDLE;
static volatile uint8_t s_custom_state_2 = HID_STATE_IDLE;

static uint8_t s_hid_custom_1_buffer[1024] = {0};
static uint8_t s_hid_custom_1_buffer_temp[1024] = {0};
static uint8_t s_hid_custom_2_buffer[1024] = {0};
static uint8_t s_hid_custom_2_buffer_temp[1024] = {0}; 

static struct ringbuffer s_hid_custom_1_ringbuffer[1];
static struct ringbuffer s_hid_custom_2_ringbuffer[1];

/* Public Variables ---------------------------------------------------------- */

/*!< custom hid report descriptor */
const uint8_t hid_custom_report_desc[HID_CUSTOM_REPORT_DESC_SIZE] = {
    /* USER CODE BEGIN 0 */
    0x06, 0x00, 0xff, // USAGE_PAGE (Vendor Defined Page 1)
    0x09, 0x01,       // USAGE (Vendor Usage 1)
    0xa1, 0x01,       // COLLECTION (Application)
    0x09, 0x01,       //   USAGE (Vendor Usage 1)
    0x15, 0x00,       //   LOGICAL_MINIMUM (0)
    0x26, 0xff, 0x00, //   LOGICAL_MAXIMUM (255)
    0x95, 0x40,       //   REPORT_COUNT (64)
    0x75, 0x08,       //   REPORT_SIZE (8)
    0x81, 0x02,       //   INPUT (Data,Var,Abs)
    /* <___________________________________________________> */
    0x09, 0x01,       //   USAGE (Vendor Usage 1)
    0x15, 0x00,       //   LOGICAL_MINIMUM (0)
    0x26, 0xff, 0x00, //   LOGICAL_MAXIMUM (255)
    0x95, 0x40,       //   REPORT_COUNT (64)
    0x75, 0x08,       //   REPORT_SIZE (8)
    0x91, 0x02,       //   OUTPUT (Data,Var,Abs)
    /* USER CODE END 0 */
    0xC0 /*     END_COLLECTION	             */
};
  
struct usbd_endpoint g_hid_custom_1_in_ep = {
    .ep_addr = USBD_IF3_AL0_EP1_ADDR,
    .ep_cb = NULL,
};

struct usbd_endpoint g_hid_custom_1_out_ep = {
    .ep_addr = USBD_IF3_AL0_EP0_ADDR,
    .ep_cb = NULL,
};

struct usbd_endpoint g_hid_custom_2_in_ep = {
    .ep_addr = USBD_IF4_AL0_EP1_ADDR,
    .ep_cb = NULL,
};

struct usbd_endpoint g_hid_custom_2_out_ep = {
    .ep_addr = USBD_IF4_AL0_EP0_ADDR,
    .ep_cb = NULL,
};

/* Private Constants --------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Initializate ringbuffer for cdc_acm.
  * @retval None
  */
void hid_ringbuffer_init(void)
{
    ringbuffer_init(s_hid_custom_1_ringbuffer, s_hid_custom_1_buffer, sizeof(s_hid_custom_1_buffer));
    ringbuffer_init(s_hid_custom_2_ringbuffer, s_hid_custom_2_buffer, sizeof(s_hid_custom_2_buffer));
    
    return;
}

/**
  * @brief            receive packet and return to USB host
  * @pre              none
  * @param[in]        none
  * @retval           none
  */
void hid_custom_1_schedual(void)
{
    uint16_t len = ringbuffer_space_len(s_hid_custom_1_ringbuffer);
    if (len >= usb_ep_out_data_avail(cmsis_dap_handle.hid_1_ep[0].ep_addr) && len)
    {
        len = usb_ep_out_data_avail(cmsis_dap_handle.hid_1_ep[0].ep_addr);
                                                                    
        if(len > 0)
        {                                                               
            usbd_read_packet(cmsis_dap_handle.hid_1_ep[0].ep_addr, s_hid_custom_1_buffer_temp, len);
            ringbuffer_put(s_hid_custom_1_ringbuffer, s_hid_custom_1_buffer_temp, len);
        }
    }
    
    len = ringbuffer_data_len(s_hid_custom_1_ringbuffer);
    if (len)
    {
        len = len > usb_ep_in_data_avail(cmsis_dap_handle.hid_1_ep[0].ep_addr) ? usb_ep_in_data_avail(cmsis_dap_handle.hid_1_ep[0].ep_addr) : len; 
        
        ringbuffer_get(s_hid_custom_1_ringbuffer, s_hid_custom_1_buffer_temp, len);
        usbd_write_packet(cmsis_dap_handle.hid_1_ep[1].ep_addr, s_hid_custom_1_buffer_temp, len);
    }
}

/**
  * @brief            receive packet and return to USB host
  * @pre              none
  * @param[in]        none
  * @retval           none
  */
void hid_custom_2_schedual(void)
{
    uint16_t len = ringbuffer_space_len(s_hid_custom_2_ringbuffer);
    if (len >= usb_ep_out_data_avail(cmsis_dap_handle.hid_2_ep[0].ep_addr) && len)
    {
        len = usb_ep_out_data_avail(cmsis_dap_handle.hid_2_ep[0].ep_addr);
        if(len > 0)
        {                                                               
            usbd_read_packet(cmsis_dap_handle.hid_2_ep[0].ep_addr, s_hid_custom_2_buffer_temp, len);
            ringbuffer_put(s_hid_custom_2_ringbuffer, s_hid_custom_2_buffer_temp, len);
        }  
    }
    
    len = ringbuffer_data_len(s_hid_custom_2_ringbuffer);
    if (len)
    {
        len = len > usb_ep_in_data_avail(cmsis_dap_handle.hid_2_ep[0].ep_addr) ? usb_ep_in_data_avail(cmsis_dap_handle.hid_2_ep[0].ep_addr) : len; 
        
        ringbuffer_get(s_hid_custom_2_ringbuffer, s_hid_custom_2_buffer_temp, len);
        usbd_write_packet(cmsis_dap_handle.hid_2_ep[1].ep_addr, s_hid_custom_2_buffer_temp, len);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
