/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          26 Oct 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "ald_conf.h" 
#include "cdc_acm_serial.h"

/* Private Macros ------------------------------------------------------------ */
#define UART_TX_PORT GPIOC
#define UART_TX_PIN  GPIO_PIN_0
#define UART_RX_PORT GPIOC
#define UART_RX_PIN  GPIO_PIN_1

/* Private Variables --------------------------------------------------------- */
static uart_handle_t g_uart_handle;
static volatile uint8_t cdc_acm_configured = 0;
static volatile struct cdc_line_coding g_line_code = {0};
static uint8_t cdc_acm_read_buffer[2048] = {0};
static uint8_t cdc_acm_write_buffer[2048] = {0};

/* Public Variables ---------------------------------------------------------- */
struct ringbuffer cdc_acm_out_ringbuffer[1];
struct ringbuffer cdc_acm_in_ringbuffer[1];

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate ringbuffer for cdc_acm.
  * @retval None
  */
static void cdc_acm_ringbuffer_init(void)
{
    ringbuffer_init(cdc_acm_out_ringbuffer, cdc_acm_read_buffer, sizeof(cdc_acm_read_buffer));
    ringbuffer_init(cdc_acm_in_ringbuffer, cdc_acm_write_buffer, sizeof(cdc_acm_write_buffer));
    
    return;
}

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
static void init_uart_pin(void)
{
    gpio_init_t gpio_init;
    memset(&gpio_init, 0x00, sizeof(gpio_init_t));

    /* Initialize tx pin: PC0 */
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin: PC1 */
    gpio_init.mode = GPIO_MODE_INPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);

    return;
}

/**
  * @brief  cdc acm serial uart param init function
  * @retval Status.
  */
static void cdc_acm_serial_param_init(struct cdc_line_coding *line_coding)
{
    struct cdc_line_coding uart_param_line_code[1];
    
    memset(&g_uart_handle, 0x00, sizeof(uart_handle_t));
    memcpy(uart_param_line_code, line_coding, sizeof(struct cdc_line_coding));
    switch (uart_param_line_code->bDataBits)
    {
        case 5:
            uart_param_line_code->bDataBits = 3;
            break;                 
        case 6:
            uart_param_line_code->bDataBits = 2;
            break;
        case 7:
            uart_param_line_code->bDataBits = 1;
            break;                 
        case 8:
            uart_param_line_code->bDataBits = 0;
            break;
        case 16:
            break;
        default:
            break;
    }
    
    switch (uart_param_line_code->bParityType)
    {
        case 0:
            uart_param_line_code->bParityType = 0;
            break;                 
        case 1:
            uart_param_line_code->bParityType = 1;
            break;
        case 2:
            uart_param_line_code->bParityType = 3;
            break;                 
        case 3:
            break;
        case 4:
            break;
        default:
            break;
    }
    
    switch (uart_param_line_code->bCharFormat)
    {
        case 0:
            uart_param_line_code->bCharFormat = 0;
            break;                 
        case 1:
            uart_param_line_code->bCharFormat = 1;
            break;
        case 2:
            uart_param_line_code->bCharFormat = 1;
            break;  
        default:
            break;
    }
    
    /* Initialize uart */
    g_uart_handle.init.baud        = uart_param_line_code->dwDTERate;
    g_uart_handle.init.word_length = uart_param_line_code->bDataBits;
    g_uart_handle.init.stop_bits   = uart_param_line_code->bCharFormat;
    g_uart_handle.init.parity      = uart_param_line_code->bParityType;
    g_uart_handle.perh             = UART1;
    g_uart_handle.init.mode        = UART_MODE_UART;
    g_uart_handle.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    g_uart_handle.rx_cplt_cbk      = NULL;
    g_uart_handle.tx_cplt_cbk      = NULL;
    ald_uart_init(&g_uart_handle);

    ald_uart_rx_fifo_config(&g_uart_handle, UART_RXFIFO_1BYTE);
    ald_uart_tx_fifo_config(&g_uart_handle, UART_TXFIFO_EMPTY);

    return;
}

/**
  * @brief  Base initialization for uart module.
  * @retval 1 indicates configuration done, else haven't configured.
  */
uint8_t cdc_acm_is_configured(void)
{    
    return cdc_acm_configured;
}

/**
  * @brief  Base initialization for uart module.
  * @retval None
  */
void cdc_acm_serial_init(void)
{
    /* Initialize ALD */
    ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_UART1, ENABLE);
    ald_mcu_irq_config(UART1_IRQn, 3, 3, ENABLE);

    /* Initialize pin */
    init_uart_pin();
    
    /* Initialize ringbuffer for uart */
    cdc_acm_ringbuffer_init();
    
    return;
}

/**
  * @brief  Redefine the function usbd_cdc_acm_set_line_coding for cdc_acm according to usbd_cdc.h.
  * @retval None
  */
void usbd_cdc_acm_set_line_coding(uint8_t intf, struct cdc_line_coding *line_coding)
{
    if (intf == 0)
    {
        memcpy((void *)(&g_line_code), line_coding, sizeof(struct cdc_line_coding));
        if ((line_coding->bCharFormat == 0 || line_coding->bCharFormat == 1 || line_coding->bCharFormat == 2) &&
            (line_coding->bParityType == 0 || line_coding->bParityType == 1 || line_coding->bParityType == 2) &&
            ((line_coding->bDataBits >= 5 && line_coding->bDataBits <= 8) || line_coding->bDataBits == 16) &&
            (line_coding->dwDTERate > 0)){
            cdc_acm_serial_param_init(line_coding);
            cdc_acm_configured = 1;            
        }
    }
}

/**
  * @brief  Redefine the function usbd_cdc_acm_get_line_coding for cdc_acm according to usbd_cdc.h.
  * @retval None
  */
void usbd_cdc_acm_get_line_coding(uint8_t intf, struct cdc_line_coding *line_coding)
{
    if (intf == 0)
    {
        memcpy(line_coding, (void *)(&g_line_code), sizeof(struct cdc_line_coding));
    }
}

/**
  * @brief  cdc uart send function
  * @retval Status.
  */
void cdc_acm_serial_xfer(void)
{
    uint16_t length = ringbuffer_data_len(cdc_acm_out_ringbuffer); 
    uint8_t data;
    
    if (length)
    {
        while (!ald_uart_get_status(&g_uart_handle, UART_STATUS_TFFULL))
        {
            ringbuffer_get(cdc_acm_out_ringbuffer, &data, 1);
            UART1->TXBUF = data;

            if (ringbuffer_data_len(cdc_acm_out_ringbuffer) == 0)
                break;
        }
    }

    return;
}

/**
  * @brief  cdc uart recv function
  * @retval Status.
  */
void cdc_acm_serial_recv(void)
{
    if (ringbuffer_space_len(cdc_acm_in_ringbuffer)){
        if (!ald_uart_get_it_status(&g_uart_handle, UART_IT_RFTH)){
            ald_uart_interrupt_config(&g_uart_handle, UART_IT_RFTH, ENABLE);
        }
    }
    
    return;
}

/**
  * @brief  UART1 handler
  * @retval None
  */      
void UART1_Handler()
{
    uint8_t data;

    if (ald_uart_get_mask_flag_status(&g_uart_handle, UART_IF_RFTH))
    {
        ald_uart_clear_flag_status(&g_uart_handle, UART_IF_RFTH);

        while (!ald_uart_get_status(&g_uart_handle, UART_STATUS_RFEMPTY))
        {
            data = UART1->RXBUF;
            if (!ringbuffer_put(cdc_acm_in_ringbuffer, &data, 1))
            {
                ald_uart_interrupt_config(&g_uart_handle, UART_IT_RFTH, DISABLE);
                SET_BIT(UART1->FCON, UART_FCON_RFRST_MSK);
                break;
            }
        }
    }
}


/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
