/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 */
#ifndef RINGBUFFER_H__
#define RINGBUFFER_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdint.h>
#include <string.h>

#ifdef USE_ASSERT
#define ringbuffer_assert_param(x)  \
do {                                \
    if (!(x)) {                     \
        while (1);                  \
    }                               \
} while (0)                     
#else
#define ringbuffer_assert_param(x)
#endif

/* ring buffer */
struct ringbuffer
{
    uint8_t *buffer_ptr;
    uint16_t read_mirror : 1;
    uint16_t read_index : 15;
    uint16_t write_mirror : 1;
    uint16_t write_index : 15;
    /* as we use msb of index as mirror bit, the size should be signed and
     * could only be positive. */
    int16_t buffer_size;
};

typedef enum ringbuffer_state
{
    RINGBUFFER_EMPTY,
    RINGBUFFER_FULL,
    /* half full is neither full nor empty */
    RINGBUFFER_HALFFULL,
}enum_ringbuffer_state;

/**
 * RingBuffer for DeviceDriver
 *
 * Please note that the ring buffer implementation of RT-Thread
 * has no thread wait or resume feature.
 */
void ringbuffer_reset(struct ringbuffer *rb);
unsigned long ringbuffer_init(struct ringbuffer *rb, uint8_t *pool, int16_t size);
unsigned long ringbuffer_put(struct ringbuffer *rb, const uint8_t *ptr, uint16_t length);
unsigned long ringbuffer_put_force(struct ringbuffer *rb, const uint8_t *ptr, uint16_t length);
unsigned long ringbuffer_putchar(struct ringbuffer *rb, const uint8_t ch);
unsigned long ringbuffer_putchar_force(struct ringbuffer *rb, const uint8_t ch);
unsigned long ringbuffer_get(struct ringbuffer *rb, uint8_t *ptr, uint16_t length);
unsigned long ringbuffer_peak(struct ringbuffer *rb, uint8_t **ptr);
unsigned long ringbuffer_getchar(struct ringbuffer *rb, uint8_t *ch);
unsigned long ringbuffer_data_len(struct ringbuffer *rb);

#ifdef RINGBUFFER_USING_HEAP
struct ringbuffer* rt_ringbuffer_create(uint16_t length);
void ringbuffer_destroy(struct ringbuffer *rb);
#endif

inline uint16_t ringbuffer_get_size(struct ringbuffer *rb)
{
    ringbuffer_assert_param(rb != NULL);
    return rb->buffer_size;
}

/** return the size of empty space in rb */
#define ringbuffer_space_len(rb) ((rb)->buffer_size - ringbuffer_data_len(rb))


#ifdef __cplusplus
}
#endif

#endif
