/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    22 Feb  2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          22 Feb  2023    shicc           version:1.0.1
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "iap_rom.h"

/* Public Variables ---------------------------------------------------------- */
typedef  void (*FunVoidType)(void);


/* Public Function ---------------------------------------------------------- */


/* Private Macros ------------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */
static md_uart_init_t s_uart_init;

/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */
void gpio_init(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief:  Init gpio for led1.
  * @param:  None
  * @retval: None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize led1 pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_config);

    md_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}
/**
  * @brief  init pins used fot uart communication.
  * @param  None
  * @retval None
  */
void uart0_pins_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize tx pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &gpio_config);

    /* Initialize rx pin */
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &gpio_config);

    return;
}

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    SYSCFG_UNLOCK();
    md_rmu_enable_gpio_reset();
    md_rmu_enable_uart0_reset();
    SYSCFG_LOCK();
}

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{ 
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;

    __disable_irq();

    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT)
    {
        addr = BOOT_ADDR;
        WRITE_REG(MSC->FLASHKEY, 0x8ACE0246);
        WRITE_REG(MSC->FLASHKEY, 0x9BDF1357);
        IAPROM_PAGE_ERASE(CRC_CAL_PAGE_ADDR);
        WRITE_REG(MSC->FLASHKEY, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
    }

    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    md_cmu_disable_perh_all();
    SYSCFG_LOCK();

    /* disable all peripherals which may cause an interrupt,and clear all possible undisposed interrupt flag */
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICER[1] = 0xFFFFFFFF;
    NVIC->ICER[2] = 0xFFFFFFFF;
    NVIC->ICER[3] = 0xFFFFFFFF;
    NVIC->ICER[4] = 0xFFFFFFFF;
    NVIC->ICER[5] = 0xFFFFFFFF;
    NVIC->ICER[6] = 0xFFFFFFFF;
    NVIC->ICER[7] = 0xFFFFFFFF;

    NVIC->ICPR[0] = 0xFFFFFFFF;
    NVIC->ICPR[1] = 0xFFFFFFFF;
    NVIC->ICPR[2] = 0xFFFFFFFF;
    NVIC->ICPR[3] = 0xFFFFFFFF;
    NVIC->ICPR[4] = 0xFFFFFFFF;
    NVIC->ICPR[5] = 0xFFFFFFFF;
    NVIC->ICPR[6] = 0xFFFFFFFF;
    NVIC->ICPR[7] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    SYSCFG_UNLOCK();
    if(para == GO_APP)
        SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    else if(para == GO_BOOT)
        SYSCFG_BOOTFLASH_MAPPING_ENABLE();
    SYSCFG_LOCK();

    /* interrupt vector remap */
    SCB->VTOR = addr & 0xFFFFFF00;

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;

    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));

    /* jump to app/boot flash */
    JumpToApplication();
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    
    /* init gpio for led */
    gpio_init();

    /* Init communication parameters */
    uart0_pins_init();
    md_uart_init_struct(&s_uart_init);
    s_uart_init.parity = MD_UART_PARITY_EVEN;
    md_uart_init(UART0, &s_uart_init);

    md_uart_set_abrmod(UART0, 1);
    md_uart_enable_abr(UART0);

    md_uart_enable_it_rfth(UART0);

    /* Config uart0 interrupt */
    NVIC_SetPriority(UART0_IRQn, 0x01);
    NVIC_EnableIRQ(UART0_IRQn);

    while (1)
    {
        if(g_flag)
        {
            g_flag = 0;

            if(g_rx_buf[1] == 0xA1 && g_rx_buf[2] == 0x5E && g_rx_buf[3] == 0xFF
            && g_rx_buf[4] == 0xFF && g_rx_buf[5] == 0xFF && g_rx_buf[6] == 0xFD
            && g_rx_buf[7] == 0xFA)
            {
                md_uart_set_send_data8(UART0, POS_ACK);
                while(!md_uart_is_active_it_tbc(UART0));
                fsm_go(GO_BOOT);
            }
            else
            {
                md_uart_set_send_data8(UART0, NEG_ACK);
                while(!md_uart_is_active_it_tbc(UART0));
            }
        }

        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(250);
    }
}


/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
