/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    biyq            the first version
 *          22 Feb  2023    shicc           version:1.0.1
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"
#include "iap_rom.h"

/* Public Variables ---------------------------------------------------------- */


/* Public Function ---------------------------------------------------------- */


/* Private Macros ------------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */

static md_spi_init_t s_spi_init;


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */
typedef  void (*FunVoidType)(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    SYSCFG_UNLOCK();
    md_rmu_enable_gpio_reset();
    md_rmu_enable_spi1_reset();
    SYSCFG_LOCK();
}

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;

    __disable_irq();

    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
        WRITE_REG(MSC->FLASHKEY, 0x8ACE0246);
        WRITE_REG(MSC->FLASHKEY, 0x9BDF1357);
        IAPROM_PAGE_ERASE(CRC_CAL_PAGE_ADDR);
        WRITE_REG(MSC->FLASHKEY, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
    }

    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    md_cmu_disable_perh_all();
    SYSCFG_LOCK();

    /* disable all peripherals which may cause an interrupt,
        and clear all possible undisposed interrupt flag */
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICER[1] = 0xFFFFFFFF;
    NVIC->ICER[2] = 0xFFFFFFFF;
    NVIC->ICER[3] = 0xFFFFFFFF;
    NVIC->ICER[4] = 0xFFFFFFFF;
    NVIC->ICER[5] = 0xFFFFFFFF;
    NVIC->ICER[6] = 0xFFFFFFFF;
    NVIC->ICER[7] = 0xFFFFFFFF;

    NVIC->ICPR[0] = 0xFFFFFFFF;
    NVIC->ICPR[1] = 0xFFFFFFFF;
    NVIC->ICPR[2] = 0xFFFFFFFF;
    NVIC->ICPR[3] = 0xFFFFFFFF;
    NVIC->ICPR[4] = 0xFFFFFFFF;
    NVIC->ICPR[5] = 0xFFFFFFFF;
    NVIC->ICPR[6] = 0xFFFFFFFF;
    NVIC->ICPR[7] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    SYSCFG_UNLOCK();
    if(para == GO_APP)
        SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    else if(para == GO_BOOT)
        SYSCFG_BOOTFLASH_MAPPING_ENABLE();

    SYSCFG_LOCK();
        
    /* interrupt vector remap */
    SCB->VTOR = addr & 0xFFFFFF00;

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;

    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));

    /* jump to app/boot flash */
    JumpToApplication();

    return;
}

/**
  * @brief:  Init gpio for led.
  * @param:  None
  * @retval: None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize led1 pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_config);

    md_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}
/**
  * @brief  init pins used fot spi communication.
  * @param  None
  * @retval None
  */
void spi1_pins_init(void)
{
    md_gpio_init_t gpio_config;

    /*From machine initialization, NSS pin configured as input mode*/
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.pupd  = MD_GPIO_PUSH_DOWN;
    gpio_config.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIOC, MD_GPIO_PIN_0, &gpio_config);

    /*From machine initialization, SCK pin configuration for the input mode*/
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.pupd  = MD_GPIO_PUSH_DOWN;
    gpio_config.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIOC, MD_GPIO_PIN_1, &gpio_config);

    /*From machine initialization MISO pins, configured to output mode*/
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIOC, MD_GPIO_PIN_2, &gpio_config);

    /*From machine initialization, MOSI pin configured as input mode*/
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.pupd  = MD_GPIO_PUSH_DOWN;
    gpio_config.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIOC, MD_GPIO_PIN_3, &gpio_config);

    return;
}


/* Public Variables ---------------------------------------------------------- */


/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* Init gpio for led */
    gpio_init();

    /* Init communication parameters */
    spi1_pins_init();
    /* master mode, clock high in free state, sample in first edge and disable sw manage slaver */
    md_spi_struct_init(&s_spi_init);
    /* change to slaver mode, receive only in double lines simplex transfer mode */
    s_spi_init.mode = MD_SPI_MODE_SLAVER;
    s_spi_init.phase = MD_SPI_CPHA_SECOND;
    md_spi_init(&s_spi_init);

    /* Config uart0 interrupt */
    NVIC_SetPriority(SPI1_I2S1_IRQn, 0x01);
    NVIC_EnableIRQ(SPI1_I2S1_IRQn);

    md_spi_enable_it_rxth(SPI1);

    while (1)
    {
        if (g_flag)
        {
            g_flag = 0;

            fsm_go(GO_BOOT);
        }

        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(250);
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
