/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"
#include "iap_rom.h"

/* Public Variables ---------------------------------------------------------- */

/* Public Function ---------------------------------------------------------- */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables--------------------------------------------------------- */
static md_i2c_init_t s_i2c_init;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */
void gpio_init(void);
typedef void (*FunVoidType)(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    SYSCFG_UNLOCK();
    md_rmu_enable_gpio_reset();
    md_rmu_enable_i2c1_reset();
    SYSCFG_LOCK();
}

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    uint32_t i;
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;
    
    for (i = 0; i < 40; i++)
        md_delay_us(500);

    __disable_irq();
    
    if (para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if (para == GO_BOOT)
    {
        addr = BOOT_ADDR;
        WRITE_REG(MSC->FLASHKEY, 0x8ACE0246);
        WRITE_REG(MSC->FLASHKEY, 0x9BDF1357);
        IAPROM_PAGE_ERASE(CRC_CAL_PAGE_ADDR);
        WRITE_REG(MSC->FLASHKEY, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
    }

    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    md_cmu_disable_perh_all();
    SYSCFG_LOCK();

    /* disable all peripherals which may cause an interrupt,
        and clear all possible undisposed interrupt flag */
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICER[1] = 0xFFFFFFFF;
    NVIC->ICER[2] = 0xFFFFFFFF;
    NVIC->ICER[3] = 0xFFFFFFFF;
    NVIC->ICER[4] = 0xFFFFFFFF;
    NVIC->ICER[5] = 0xFFFFFFFF;
    NVIC->ICER[6] = 0xFFFFFFFF;
    NVIC->ICER[7] = 0xFFFFFFFF;

    NVIC->ICPR[0] = 0xFFFFFFFF;
    NVIC->ICPR[1] = 0xFFFFFFFF;
    NVIC->ICPR[2] = 0xFFFFFFFF;
    NVIC->ICPR[3] = 0xFFFFFFFF;
    NVIC->ICPR[4] = 0xFFFFFFFF;
    NVIC->ICPR[5] = 0xFFFFFFFF;
    NVIC->ICPR[6] = 0xFFFFFFFF;
    NVIC->ICPR[7] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    SYSCFG_UNLOCK();
    if (para == GO_APP)
        SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    else if (para == GO_BOOT)
        SYSCFG_BOOTFLASH_MAPPING_ENABLE();
    SYSCFG_LOCK();

    /* interrupt vector remap */
    SCB->VTOR = addr & 0xFFFFFF00;

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;

    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));

    /* jump to app application */
    JumpToApplication();

    return;
}

/**
  * @brief:  Init gpio for led1.
  * @param:  None
  * @retval: None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize led1 pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_config);

    md_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}

/**
  * @brief  init pins used fot uart communication.
  * @param  None
  * @retval None
  */
void i2c1_pins_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize scl pin */
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_OPEN_DRAIN;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_5;
    md_gpio_init(I2C_SCL_PORT, I2C_SCL_PIN, &gpio_config);

    /* Initialize sda pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_OPEN_DRAIN;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_5;
    md_gpio_init(I2C_SDA_PORT, I2C_SDA_PIN, &gpio_config);

    return;
}


/* Public Variables ---------------------------------------------------------- */

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* init gpio for led1 */
    gpio_init();

    /* Init communication parameters */
    i2c1_pins_init();
    /* slave address: 0x2D, 7 bits address, 100K, SCL with auto stretch, rx_fifo: 0 byte, tx_fifo: 1 byte */
    s_i2c_init.clk_speed    = MD_I2C_STANDARD_MODE_MAX_CLK;
    s_i2c_init.addr_mode    = MD_I2C_ADDR_7BIT;
    s_i2c_init.own_addr2    = 0x0;
    s_i2c_init.general_call = MD_I2C_GENERALCALL_DISABLE;
    s_i2c_init.module       = MD_I2C_MODULE_SLAVE;
    s_i2c_init.dual_addr    = MD_I2C_DUALADDR_ENABLE;
    s_i2c_init.no_stretch   = MD_I2C_NOSTRETCH_DISABLE;
    s_i2c_init.own_addr1    = SLAVE_ADDR << 1;
    md_i2c_init(COMMUNICATION_TYPE, &s_i2c_init);

    /* Config i2c1 interrupt */
    NVIC_SetPriority(I2C1_EV_IRQn, 0x01);
    NVIC_EnableIRQ(I2C1_EV_IRQn);
    NVIC_SetPriority(I2C1_ERR_IRQn, 0x01);
    NVIC_EnableIRQ(I2C1_ERR_IRQn);

    md_i2c_enable_it_addr(COMMUNICATION_TYPE);
    md_i2c_enable_it_stop(COMMUNICATION_TYPE);
    md_i2c_enable_it_rxth(COMMUNICATION_TYPE);

    while (1)
    {
        if(g_flag)
        {
            g_flag = 0;

            fsm_go(GO_BOOT);
        }

        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(250);
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
