/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    biyq            the first version
 *          22 Feb  2023    shicc           version:1.0.1
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"
#include "iap_rom.h"

/* Public Variables ---------------------------------------------------------- */
can_handle_t g_can_handle;
can_rx_msg_t g_rx_msg;   /*CAN RX msg obj */
uint16_t g_frame_timer;

/* Public Function ---------------------------------------------------------- */


/* Private Macros ------------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */
void gpio_init(void);
typedef  void (*FunVoidType)(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    ald_rmu_reset_periperal(RMU_PERH_GPIO);
    ald_rmu_reset_periperal(RMU_PERH_CAN0);
    ald_rmu_reset_periperal(RMU_PERH_CRC);
    ald_rmu_reset_periperal(RMU_PERH_IWDT);

    return;
}

/**
  * @brief  define the function used to jump to app/boot program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;
    
    __disable_irq();  
    
    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
        WRITE_REG(MSC->FLASHKEY, 0x8ACE0246);
        WRITE_REG(MSC->FLASHKEY, 0x9BDF1357);
        IAPROM_PAGE_ERASE(CRC_CAL_PAGE_ADDR);
        WRITE_REG(MSC->FLASHKEY, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
    }

    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    ald_cmu_perh_clock_config(CMU_PERH_ALL, DISABLE);
    SYSCFG_LOCK();

    /* disable all peripherals which may cause an interrupt,
        and clear all possible undisposed interrupt flag */
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICER[1] = 0xFFFFFFFF;
    NVIC->ICER[2] = 0xFFFFFFFF;
    NVIC->ICER[3] = 0xFFFFFFFF;
    NVIC->ICER[4] = 0xFFFFFFFF;
    NVIC->ICER[5] = 0xFFFFFFFF;
    NVIC->ICER[6] = 0xFFFFFFFF;
    NVIC->ICER[7] = 0xFFFFFFFF;

    NVIC->ICPR[0] = 0xFFFFFFFF;
    NVIC->ICPR[1] = 0xFFFFFFFF;
    NVIC->ICPR[2] = 0xFFFFFFFF;
    NVIC->ICPR[3] = 0xFFFFFFFF;
    NVIC->ICPR[4] = 0xFFFFFFFF;
    NVIC->ICPR[5] = 0xFFFFFFFF;
    NVIC->ICPR[6] = 0xFFFFFFFF;
    NVIC->ICPR[7] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    SYSCFG_UNLOCK();
    if(para == GO_APP)
        SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    else if(para == GO_BOOT)
        SYSCFG_BOOTFLASH_MAPPING_ENABLE();
    SYSCFG_LOCK();

    /* interrupt vector remap */
    SCB->VTOR = addr & 0xFFFFFF00;

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;

    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));

    /* jump to app/boot application */
    JumpToApplication();

    return;
}

/**
  * @brief:  Init gpio for led1.
  * @param:  None
  * @retval: None
  */
void gpio_init(void)
{
     gpio_init_t gpio_config;

    /* Initialize led1 pin */
    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = GPIO_OUT_DRIVE_6;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_CMOS;
    gpio_config.func  = GPIO_FUNC_1;
    ald_gpio_init(LED_PORT, LED_PIN, &gpio_config);

    ald_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}
/**
  * @brief  init pins used fot can communication.
  * @param  None
  * @retval None
  */
void can_pins_init(void)
{
    gpio_init_t gpio_config;

    /* Initialize CAN_TX pin */  
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.mode  = GPIO_MODE_OUTPUT; 
    gpio_config.nodrv = GPIO_OUT_DRIVE_6;
    gpio_config.podrv = GPIO_OUT_DRIVE_6;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_9, &gpio_config);

    /* Initialize CAN_RX pin */
    gpio_config.mode  = GPIO_MODE_INPUT;
    ald_gpio_init(GPIOB, GPIO_PIN_8, &gpio_config);

    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to can_handle_t structure.
  * @retval None.
  */
void can_send_complete(can_handle_t *arg)
{
    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to can_handle_t structure.
  * @param  num: Index of the RxFIFO.
  * @retval None.
  */
void can_recv_complete(can_handle_t *arg, can_rx_fifo_t num)
{
    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to can_handle_t structure.
  * @retval None.
  */
void can_error(can_handle_t *arg)
{
    return;
}

/**
  * @brief  Initializing can function.
  * @retval None.
  */
void can_init(void)
{
    can_filter_t can_filter_config;

    g_can_handle.perh        = CAN0;
    g_can_handle.init.mode   = CAN_MODE_NORMAL;
    g_can_handle.init.psc    = 8;
    g_can_handle.init.sjw    = CAN_SJW_1;
    g_can_handle.init.seg1   = CAN_SEG1_7;
    g_can_handle.init.seg2   = CAN_SEG2_4;
    g_can_handle.init.ttcm   = DISABLE;
    g_can_handle.init.abom   = DISABLE;
    g_can_handle.init.awk    = DISABLE;
    g_can_handle.init.artx   = DISABLE;
    g_can_handle.init.rfom   = DISABLE;
    g_can_handle.init.txmp   = DISABLE;
    g_can_handle.tx_cplt_cbk = can_send_complete;
    g_can_handle.rx_cplt_cbk = can_recv_complete;
    g_can_handle.error_cbk   = can_error;
    ald_can_init(&g_can_handle);                   /* init CAN module */

    can_filter_config.fifo         = CAN_FILTER_FIFO0;
    can_filter_config.mode         = CAN_FILTER_MODE_MASK;
    can_filter_config.scale        = CAN_FILTER_SCALE_32;
    can_filter_config.number       = 0;
    can_filter_config.id_high      = MASKID << 5;    /* frame ID 0x003 */
    can_filter_config.id_low       = 0;
    can_filter_config.mask_id_high = 0xFF00;    /*Mask the high register and only receive messages with identifiers 0x5A0 to 0x5A7*/
    can_filter_config.mask_id_low  = 0x0000;
    can_filter_config.active       = ENABLE;
    ald_can_filter_config(&g_can_handle, &can_filter_config);           /* config and enable CAN flt0 */

    ald_mcu_irq_config(CAN0_RX0_IRQn, 3, 3, ENABLE);    /* enable CAN RX interrupt */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
 int main(void)
{
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* init gpio for led */
    gpio_init();

    /* Config can */
    can_pins_init();
    can_init();

    /* enable commnication to receive */
    ald_can_recv_by_it(&g_can_handle, CAN_RX_FIFO0, &g_rx_msg);

    while (1)
    {
        if(g_flag)
        {
            g_flag = 0;

            fsm_go(GO_BOOT);
        }

        ald_gpio_toggle_pin(LED_PORT, LED_PIN);
        ald_delay_ms(250);
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
