/**********************************************************************************
 *
 * @file    irq.c
 * @brief   Interrupt handler
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Aug 2021      biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include <stdint.h>
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
static can_tx_msg_t s_tx_msg;   /*CAN TX msg obj */

/* Public Variables ---------------------------------------------------------- */
uint8_t g_rx_buf[64];
uint8_t g_tx_buf = NEG_ACK;
uint16_t g_rx_len = sizeof(g_rx_buf);
uint16_t g_rx_i = 0U;
uint8_t g_flag = 0U;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  NMI IRQ handler
  * @retval None
  */
void NMI_Handler(void)
{
    /* Added Emergency operation */
    return;
}

/**
  * @brief  Hardfault IRQ handler
  * @retval None
  */
void HardFault_Handler(void)
{
    /* Added debug information */
    while (1)
        ;
}

/**
  * @brief  MemManage IRQ handler
  * @retval None
  */
void MemManage_Handler(void)
{
    /* Added debug information */
    while (1)
        ;
}

/**
  * @brief  BusFault IRQ handler
  * @retval None
  */
void BusFault_Handler(void)
{
    /* Added debug information */
    while (1)
        ;
}

/**
  * @brief  UsageFault IRQ handler
  * @retval None
  */
void UsageFault_Handler(void)
{
    /* Added debug information */
    while (1)
        ;
}

/**
  * @brief  Supervisor Call IRQ handler
  * @retval None
  */
void SVC_Handler(void)
{
    /* Added system callback */
    return;
}

/**
  * @brief  Debug Monitor IRQ handler
  * @retval None
  */
void DebugMon_Handler(void)
{
    /* Added debug operation */
    return;
}

/**
  * @brief  PendSV IRQ handler
  * @retval None
  */
void PendSV_Handler(void)
{
    /* Added thread switching operation */
    return;
}

/**
  * @brief  SysTick IRQ handler
  * @retval None
  */
void SysTick_Handler(void)
{
    ald_inc_tick();

    /* the time interval of received adjacent bytes in the same frame is more than expected, reset rx FIFO */
    if (g_frame_timer != 0)
    {
        g_frame_timer--;

        if (g_frame_timer == 0)
            g_rx_i = 0;
    }

    return;
}

/**
  * @brief  CMU IRQ#4 handler
  * @retval None
  */
void CMU_Handler(void)
{
    ald_cmu_irq_handler();
    return;
}
#ifdef ALD_DMA
/**
  * @brief  DMA IRQ#66 handler
  * @retval None
  */
void DMA_Handler(void)
{
    ald_dma_irq_handler();
}
#endif  /* ALD_DMA */

/**
  * @brief  Receives a correct CAN frame using interrupt.
  * @param  hperh:  Pointer to a can_handle_t structure.
  * @param  num: Specify the FIFO number
  * @retval Status, see ald_status_t.
  */
static ald_status_t can_recv_by_it(can_handle_t *hperh, can_rx_fifo_t num)
{
    uint32_t stid, exid;
    can_rx_msg_t *_msg;

    stid = READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_STDID_MSK, CAN_RXF0ID_STDID_POSS);
    exid = READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_EXID_MSK, CAN_RXF0ID_EXID_POSS);

    _msg = num == CAN_RX_FIFO0 ? hperh->rx0_msg : hperh->rx1_msg;
    _msg->type = (can_id_type_t)READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_IDE_MSK, CAN_RXF0ID_IDE_POS);

    if (_msg->type == CAN_ID_STD)
        _msg->std = stid;
    else
        _msg->ext = (stid << 18) | exid;

    _msg->rtr     = (can_remote_req_t)READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_RTR_MSK, CAN_RXF0ID_RTR_POS);
    _msg->len     = READ_BITS(hperh->perh->RxFIFO[num].RXFINF, CAN_RXF0INF_DLEN_MSK, CAN_RXF0INF_DLEN_POSS);
    _msg->fmi     = READ_BITS(hperh->perh->RxFIFO[num].RXFINF, CAN_RXF0INF_FLTIDX_MSK, CAN_RXF0INF_FLTIDX_POSS);
    _msg->data[0] = hperh->perh->RxFIFO[num].RXFDL & 0xFF;
    _msg->data[1] = (hperh->perh->RxFIFO[num].RXFDL >> 8) & 0xFF;
    _msg->data[2] = (hperh->perh->RxFIFO[num].RXFDL >> 16) & 0xFF;
    _msg->data[3] = (hperh->perh->RxFIFO[num].RXFDL >> 24) & 0xFF;
    _msg->data[4] = hperh->perh->RxFIFO[num].RXFDH & 0xFF;
    _msg->data[5] = (hperh->perh->RxFIFO[num].RXFDH >> 8) & 0xFF;
    _msg->data[6] = (hperh->perh->RxFIFO[num].RXFDH >> 16) & 0xFF;
    _msg->data[7] = (hperh->perh->RxFIFO[num].RXFDH >> 24) & 0xFF;

    /* can_rx_fifo_release */
    SET_BIT(hperh->perh->RXF0, CAN_RXF0_FREE_MSK);

    return OK;
}

/**
  * @brief  CAN TX handler
  * @retval None
  */
void CAN0_TX_Handler(void)
{
    /* Handle can interrupt */
    return;
}

/**
  * @brief  CAN RX0 handler
  * @retval None
  */
void CAN0_RX0_Handler(void)
{
    if (ald_can_get_it_status(&g_can_handle, CAN_IT_FP0) && (CAN_RX_MSG_PENDING(&g_can_handle, CAN_RX_FIFO0) != 0))
    {
        /* enable rxfifo0 to receive msg via interrupt */
        can_recv_by_it(&g_can_handle, CAN_RX_FIFO0);

        /* assign timer to default when MCU receive each byte */
        g_frame_timer = FRAME_INTERVAL;

        if (g_rx_msg.data[0] == 0x3F)
        {
            if (g_rx_msg.data[1] == 0xA1 && g_rx_msg.data[2] == 0x5E && g_rx_msg.data[3] == 0xFF
                    && g_rx_msg.data[4] == 0xFF && g_rx_msg.data[5] == 0xFF && g_rx_msg.data[6] == 0xFD
                    && g_rx_msg.data[7] == 0xFA)
                g_tx_buf = POS_ACK;
            else
                g_tx_buf = NEG_ACK;

            s_tx_msg.rtr  = g_rx_msg.rtr;      /*发送数据帧*/
            s_tx_msg.type = g_rx_msg.type;     /*标准格式*/
            s_tx_msg.std  = g_rx_msg.std;      /*标准消息标识符*/
            s_tx_msg.ext  = g_rx_msg.ext;      /*扩展消息标识符*/
            s_tx_msg.len  = g_rx_msg.len;      /*发送的数据长度*/
            s_tx_msg.data[0] = g_tx_buf;       /*发送数据赋值*/
            ald_can_send(&g_can_handle, &s_tx_msg, TIMEOUT);

            if (g_tx_buf == POS_ACK)
                g_flag = 1;
            else
                g_flag = 0;
        }

        ald_can_recv_by_it(&g_can_handle, CAN_RX_FIFO0, &g_rx_msg);
    }

    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
