/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "comm_proc.h"


/* Private Macros ------------------------------------------------------------ */


/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Private Variables--------------------------------------------------------- */

/**
  *@brief enum definition of state machine procedure
  */
typedef struct comm_s
{
    uint8_t      comm_tx_buf[8];
    uint8_t      comm_rx_buf[8];
    uint32_t     comm_tx_cnt;
    uint32_t     comm_rx_cnt;
    uint32_t     comm_rx_redundant_cnt;
    uint32_t     comm_tx_size;
    uint32_t     comm_rx_size;
} comm_t;

volatile static comm_t  s_comm_handler;
static  uint8_t    *s_can_rx_buf;

static can_handle_t s_can_handle;
static can_tx_msg_t s_tx_msg;   /*CAN TX msg obj */
static can_rx_msg_t s_rx_msg;   /*CAN RX msg obj */


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  this function is designed to judge whether date the state machine wanna send has been read out by upper
            this function will always return 0, due to the fact, this function only be called via tbc interrupt when
            all expected data sent out in interrupt program
  * @param  None
  * @retval 0x00 to indicate all data expected to send has been read out by upper,
  *         0x01 to indicate process of send is in progress
  */
static fsm_result_t fsm_is_send_over(void)
{
    if (s_comm_handler.comm_tx_cnt != 0 && s_comm_handler.comm_tx_size == 0)
    {
        return PASS;
    }
    else
    {
        return FAIL;
    }
}

/**
  * @brief  this function is designed to judge whether date the state machine wanna receive has been completed
            this function will always return 0, due to the fact, this function only be called when all expected data
            received in interrupt program
  * @param  None
  * @retval 0x00 to indicate all data expected to receive has been completed,
  *         0x01 to indicate process of receive is in progress
  */
static fsm_result_t fsm_is_recv_over(void)
{
    if (s_comm_handler.comm_rx_size > 0 && s_comm_handler.comm_rx_size == s_comm_handler.comm_rx_cnt)
    {
        return PASS;
    }
    else
    {
        return FAIL;
    }
}

/**
  * @brief  define the send function.
  * @param  None
  * @retval None
  */
static void fsm_send_data(uint8_t *tx_buf, uint32_t len)
{
    s_comm_handler.comm_tx_size = len;
    s_comm_handler.comm_tx_cnt = 0;

    /* enable commnication to send */
    /* keep the send format same with the received */
    s_tx_msg.rtr  = s_rx_msg.rtr;
    s_tx_msg.type = s_rx_msg.type;
    s_tx_msg.std  = s_rx_msg.std;
    s_tx_msg.ext  = s_rx_msg.ext;

    /* copy data tx mailbox */
    while (s_comm_handler.comm_tx_size)
    {
        if (s_comm_handler.comm_tx_size > 8)
        {
            memcpy(s_tx_msg.data, (void *)(tx_buf + s_comm_handler.comm_tx_cnt), 8);
            s_comm_handler.comm_tx_size -= 8;
            s_comm_handler.comm_tx_cnt += 8;
            s_tx_msg.len  = 8;
        }
        else
        {
            memcpy(s_tx_msg.data, (void *)(tx_buf + s_comm_handler.comm_tx_cnt), s_comm_handler.comm_tx_size);
            s_tx_msg.len  = s_comm_handler.comm_tx_size;
            s_comm_handler.comm_tx_cnt += s_comm_handler.comm_tx_size;
            s_comm_handler.comm_tx_size = 0;
        }

        ald_can_send(&s_can_handle, &s_tx_msg, TIMEOUT);
    }

    fsm_send_over();

    /*   */
    return;
}

/**
  * @brief  define the receive function.
  * @param  None
  * @retval None
  */
static void fsm_recv_data(uint8_t *rx_buf, uint32_t len)
{
    s_comm_handler.comm_rx_size = len;
    s_comm_handler.comm_rx_cnt = 0;

    /* allign static variable can_rx_buf to rx_buf */
    s_can_rx_buf = rx_buf;

    if (s_comm_handler.comm_rx_redundant_cnt >= s_comm_handler.comm_rx_size)
    {
        memcpy(s_can_rx_buf, (void *)(s_comm_handler.comm_rx_buf), s_comm_handler.comm_rx_size);
        s_comm_handler.comm_rx_redundant_cnt -= s_comm_handler.comm_rx_size;
        memcpy((void *)(s_comm_handler.comm_rx_buf), (void *)(s_comm_handler.comm_rx_buf + s_comm_handler.comm_rx_size), s_comm_handler.comm_rx_redundant_cnt);
        s_comm_handler.comm_rx_cnt = s_comm_handler.comm_rx_size;

        /* received all expected data and trigger state machine to implement the next step */
        fsm_recv_over();
    }
    else
    {
        memcpy(s_can_rx_buf, (void *)(s_comm_handler.comm_rx_buf), s_comm_handler.comm_rx_redundant_cnt);
        s_comm_handler.comm_rx_size -= s_comm_handler.comm_rx_redundant_cnt;
        s_comm_handler.comm_rx_cnt = s_comm_handler.comm_rx_redundant_cnt;
        s_comm_handler.comm_rx_redundant_cnt = 0;
        memset((void *)(s_comm_handler.comm_rx_buf), 0x00, 8);

        /* enable commnication to receive */
        ald_can_recv_by_it(&s_can_handle, CAN_RX_FIFO0, &s_rx_msg);
    }

    /*   */
    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to can_handle_t structure.
  * @retval None.
  */
void can_error(can_handle_t *arg)
{
    fsm_para_init();

    return;
}

/**
  * @brief  Receives a correct CAN frame using interrupt.
  * @param  hperh:  Pointer to a can_handle_t structure.
  * @param  num: Specify the FIFO number
  * @retval Status, see ald_status_t.
  */
static ald_status_t can_recv_by_it(can_handle_t *hperh, can_rx_fifo_t num)
{
    uint32_t stid, exid;
    can_rx_msg_t *_msg;

    stid = READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_STDID_MSK, CAN_RXF0ID_STDID_POSS);
    exid = READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_EXID_MSK, CAN_RXF0ID_EXID_POSS);

    _msg = num == CAN_RX_FIFO0 ? hperh->rx0_msg : hperh->rx1_msg;
    _msg->type = (can_id_type_t)READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_IDE_MSK, CAN_RXF0ID_IDE_POS);

    if (_msg->type == CAN_ID_STD)
        _msg->std = stid;
    else
        _msg->ext = (stid << 18) | exid;

    _msg->rtr     = (can_remote_req_t)READ_BITS(hperh->perh->RxFIFO[num].RXFID, CAN_RXF0ID_RTR_MSK, CAN_RXF0ID_RTR_POS);
    _msg->len     = READ_BITS(hperh->perh->RxFIFO[num].RXFINF, CAN_RXF0INF_DLEN_MSK, CAN_RXF0INF_DLEN_POSS);
    _msg->fmi     = READ_BITS(hperh->perh->RxFIFO[num].RXFINF, CAN_RXF0INF_FLTIDX_MSK, CAN_RXF0INF_FLTIDX_POSS);
    _msg->data[0] = hperh->perh->RxFIFO[num].RXFDL & 0xFF;
    _msg->data[1] = (hperh->perh->RxFIFO[num].RXFDL >> 8) & 0xFF;
    _msg->data[2] = (hperh->perh->RxFIFO[num].RXFDL >> 16) & 0xFF;
    _msg->data[3] = (hperh->perh->RxFIFO[num].RXFDL >> 24) & 0xFF;
    _msg->data[4] = hperh->perh->RxFIFO[num].RXFDH & 0xFF;
    _msg->data[5] = (hperh->perh->RxFIFO[num].RXFDH >> 8) & 0xFF;
    _msg->data[6] = (hperh->perh->RxFIFO[num].RXFDH >> 16) & 0xFF;
    _msg->data[7] = (hperh->perh->RxFIFO[num].RXFDH >> 24) & 0xFF;

    /* can_rx_fifo_release */
    SET_BIT(hperh->perh->RXF0, CAN_RXF0_FREE_MSK);

    return OK;
}

/* Public Function ---------------------------------------------------------- */

/**
  * @brief  Initializing can function.
  * @retval None.
  */
void can_init(void)
{
    can_filter_t can_filter_config;

    s_can_handle.perh        = CAN0;
    s_can_handle.init.mode   = CAN_MODE_NORMAL;
    s_can_handle.init.psc    = 8;
    s_can_handle.init.sjw    = CAN_SJW_1;
    s_can_handle.init.seg1   = CAN_SEG1_7;
    s_can_handle.init.seg2   = CAN_SEG2_4;
    s_can_handle.init.ttcm   = DISABLE;
    s_can_handle.init.abom   = DISABLE;
    s_can_handle.init.awk    = DISABLE;
    s_can_handle.init.artx   = DISABLE;
    s_can_handle.init.rfom   = DISABLE;
    s_can_handle.init.txmp   = DISABLE;
    s_can_handle.tx_cplt_cbk = NULL;
    s_can_handle.rx_cplt_cbk = NULL;
    s_can_handle.error_cbk   = can_error;
    ald_can_init(&s_can_handle);                   /* init CAN module */

    can_filter_config.fifo         = CAN_FILTER_FIFO0;
    can_filter_config.mode         = CAN_FILTER_MODE_MASK;
    can_filter_config.scale        = CAN_FILTER_SCALE_32;
    can_filter_config.number       = 0;
    can_filter_config.id_high      = MASKID << 5;    /* frame ID 0x003 */
    can_filter_config.id_low       = 0;
    can_filter_config.mask_id_high = 0xFF00;
    can_filter_config.mask_id_low  = 0x0000;
    can_filter_config.active       = ENABLE;
    ald_can_filter_config(&s_can_handle, &can_filter_config);           /* config and enable CAN flt0 */

    ald_mcu_irq_config(CAN0_RX0_IRQn, 3, 3, ENABLE);    /* enable CAN RX interrupt */
}

/**
  * @brief  assign function pointer related to communication to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_comm_func_init(void)
{
    memset((void *)(&s_comm_handler), 0x00, sizeof(s_comm_handler));
    g_isp_data.p_send_func    = &fsm_send_data;
    g_isp_data.p_recv_func    = &fsm_recv_data;
    g_isp_data.p_is_send_over = &fsm_is_send_over;
    g_isp_data.p_is_recv_over = &fsm_is_recv_over;
}

/**
  * @brief  function to handle communication interrupt.
  * @param  None
  * @retval None
  */
void fsm_int_exec(void)
{
    if (ald_can_get_it_status(&s_can_handle, CAN_IT_FP0) && (CAN_RX_MSG_PENDING(&s_can_handle, CAN_RX_FIFO0) != 0))
    {
        /* enable rxfifo0 to receive msg via interrupt */
        can_recv_by_it(&s_can_handle, CAN_RX_FIFO0);

        if (s_comm_handler.comm_rx_size <= s_rx_msg.len)
        {
            memcpy(s_can_rx_buf + s_comm_handler.comm_rx_cnt, s_rx_msg.data, s_comm_handler.comm_rx_size);
            s_comm_handler.comm_rx_redundant_cnt = s_rx_msg.len - s_comm_handler.comm_rx_size;
            memcpy((void *)(s_comm_handler.comm_rx_buf), s_rx_msg.data + s_comm_handler.comm_rx_size, s_comm_handler.comm_rx_redundant_cnt);
            s_comm_handler.comm_rx_cnt = s_comm_handler.comm_rx_size;

            /* received all expected data and store the redudant data to s_comm_handler.comm_rx_buf */
            fsm_recv_over();
        }
        else
        {
            memcpy(s_can_rx_buf + s_comm_handler.comm_rx_cnt, s_rx_msg.data, s_rx_msg.len);
            s_comm_handler.comm_rx_cnt += s_rx_msg.len;
            s_comm_handler.comm_rx_size -= s_rx_msg.len;
        }
    }
}

/**
  * @brief  function to handle systick interrupt.
  * @param  None
  * @retval None
  */
void fsm_systick_int_exec(void)
{

    /* the time interval of received adjacent bytes in the same frame is more than expected, reset tx/rx FIFO */
    if (g_isp_data.u_frame_timer != 0)
    {
        g_isp_data.u_frame_timer--;

        if (g_isp_data.u_frame_timer == 0)
        {

            fsm_para_init();
            return;
        }
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
