/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Jun 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
  
/* Private Macros ------------------------------------------------------------ */
#define XMODE_STX   0x02    /**< header of 1k-xmode  frames */
#define XMODE_EOT   0x04    /**< frame end byte */
#define XMODE_ACK   0x06    /**< frame ACK byte */
#define XMODE_NACK  0x15    /**< frame NACK byte */
#define XMODE_CAN   0x18    /**< revocation frame transmission */
#define XMODE_EOF   0x1A    /**< padding data */
#define XMODE_CRC   0x43    /**< CRC checkout frame */
#define TIME_OUT    3000    /**< timeout(ms) */
#define TRY_CNT     10  /**< repeat try number */

#define UPDATE_FLASH_ADDR 0x44000
#define UPDATE_FLASH_SIZE 0x3C000
#define INFO_FLASH_ADDR 0x7C00

/* Private Constants --------------------------------------------------------- */

enum{
    XMODEM_STATE_START,
    XMODEM_STATE_RECEIVING,
};
enum{
    XMODEM_SUCCESS,
    XMODEM_TIMEOUT,
    XMODEM_INDEX_ERROR,
    XMODEM_CRC_ERROR,
    XMODEM_FLASH_ERROR
};


/* Private function prototypes ----------------------------------------------- */   
void crc_init(void);
void uart0_init(void);
void printf_e(const char *fmt, ...);

/* Private Variables --------------------------------------------------------- */
static uart_handle_t s_h_uart;
static crc_handle_t s_h_crc;
static xmodem_state_t s_xmod;

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  XModem send cmd
  * @retval (0)-success (-1)-error
  */
static int xmod_cmd_send(uint8_t cmd)
{
    if ((ald_uart_send(&s_h_uart, &cmd, 1, 100)) != OK)
    {
        return -1;
    }

    return 0;
}

/**
  * @brief  write data to flash at specified address.
  * @param  page: specific page.
  * @param  data: write data
  * @param  crc: data crc.
  * @retval 0-success, other value indicates failed.
  */
int write_data_to_flash(uint32_t addr, uint8_t *data, uint16_t crc)
{
    uint16_t _crc;

    if (ald_iap_program_words(addr, data, 1024, ENABLE))
        return -1;
    
    CRC_RESET(&s_h_crc);
    _crc = ald_crc_calculate(&s_h_crc, (uint8_t*)addr, 1024);

    if (_crc != crc)
        return -2;

    return 0;
}
/**
  * @brief  Receive data by 1K_XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_1K_recv()
{
    uint16_t crc, _crc;
    uint8_t soh;
    
    /* SOH */
    if ((ald_uart_recv(&s_h_uart, &soh, 1, TIME_OUT)) == TIMEOUT)
    {
        s_xmod.error = XMODEM_TIMEOUT;
        return -1;
    }

    if (soh == XMODE_EOT)
    {
        s_xmod.error = XMODEM_SUCCESS;
        return 1;
    }

    /* DATA */
    if ((ald_uart_recv(&s_h_uart, s_xmod.data, 1028, TIME_OUT)) == TIMEOUT)
    {
        s_xmod.error = XMODEM_TIMEOUT;
        return -1;
    }

    if ((s_xmod.data[0] ^ s_xmod.data[1]) != 0xFF)
    {
        s_xmod.error = XMODEM_INDEX_ERROR;
        return -2;
    }

    if (s_xmod.state==XMODEM_STATE_START)
    {
        s_xmod.idx=s_xmod.data[0];
        s_xmod.state=XMODEM_STATE_RECEIVING;
    }
    else
    {
        if (s_xmod.idx!=s_xmod.data[0])
        {
            s_xmod.error = XMODEM_INDEX_ERROR;
            return -2;
        }
    }

    CRC_RESET(&s_h_crc);
    _crc = ald_crc_calculate(&s_h_crc, &s_xmod.data[2], 1024);
    crc = (s_xmod.data[1026] << 8) | s_xmod.data[1027];

    if (crc != _crc)
    {
        s_xmod.error = XMODEM_CRC_ERROR;
        return -3;
    }

    if (write_data_to_flash(s_xmod.write_addr,&s_xmod.data[2],crc))
    {
        s_xmod.error = XMODEM_FLASH_ERROR;
        return -4;
    }
    
    s_xmod.write_addr+=1024;
    s_xmod.idx++;
    return 0;
}

/**
  * @brief  Xmodem receive app update
  * @retval 0-success 1-failed
  */
int xmodem_recv()
{
    uint8_t result=0U;
    s_xmod.state=XMODEM_STATE_START;
    s_xmod.error=XMODEM_SUCCESS;
    s_xmod.write_addr=UPDATE_FLASH_ADDR;
    s_xmod.idx=0;
    
    xmod_cmd_send(XMODE_CRC);
    while (1)
    {
        result=xmod_1K_recv();
        
        if (result==0)
        {
            /* Continue receive data */
            xmod_cmd_send(XMODE_ACK);
        }
        else if (result==1)
        {
            /* Transmision end */
            xmod_cmd_send(XMODE_ACK);
            break;
        }
        else
        {
            if (s_xmod.error!=XMODEM_TIMEOUT)
            {
                xmod_cmd_send(XMODE_CAN);
            }
            else
            {
                xmod_cmd_send(XMODE_NACK);
            }
            /* Error exit */
            break;
        }
    }
    return result;
}
/**
  * @brief  Check app flash and copy info data to last page
  * @retval 0-success, 1-error
  */
uint32_t write_update_info(void)
{
	uint32_t dcrc=0U;
	uint32_t dlen=0U;
	uint32_t*info_data;
    uint32_t result=0U;
	
	info_data = (uint32_t*)(s_xmod.write_addr-8);
	dcrc = info_data[0];
	dlen = info_data[1];
	
	if (dlen>UPDATE_FLASH_SIZE)
	{
		return 1; /*User app length invalid */
	}
	
	if (s_xmod.write_addr!=UPDATE_FLASH_ADDR+UPDATE_FLASH_SIZE)
	{
		/* We have to move app_info to the last page of app_flash */
		result|=ald_iap_program_word(INFO_FLASH_ADDR, dlen);
		result|=ald_iap_program_word(INFO_FLASH_ADDR + 4, dcrc);
	}
    return result;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	/* Enable peripheral clock */ 
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	uart0_init();
    crc_init();
    
    printf_e("Boot demo\r\n");
    printf_e(__DATE__ " " __TIME__ "\r\n");
    printf_e("Ready to receive app\r\n");
    while (1)
    {
        if (xmodem_recv()==1)
        {
            write_update_info();
            break;
        }
    }
    ald_delay_ms(1000);
    printf_e("Receive new app data ok\r\n");
    printf_e("Reboot to update\r\n");
    while (1);
}

/**
  * @brief  Initialize CRC
  * @retval None
  */
void crc_init(void)
{
    /* Initialize CRC */
    s_h_crc.perh          = CRC;
    s_h_crc.init.mode     = CRC_MODE_CCITT;
    s_h_crc.init.seed     = 0x0;
    s_h_crc.init.data_rev = DISABLE;
    s_h_crc.init.data_inv = DISABLE;
    s_h_crc.init.chs_rev  = DISABLE;
    s_h_crc.init.chs_inv  = DISABLE;
    ald_crc_init(&s_h_crc);
}
/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart0_init(void)
{
	gpio_init_t gpio;
	
	/* Initialize tx pin */
	gpio.mode  = GPIO_MODE_OUTPUT;
	gpio.odos  = GPIO_PUSH_PULL;
	gpio.pupd  = GPIO_PUSH_UP;
	gpio.podrv = GPIO_OUT_DRIVE_6;
	gpio.nodrv = GPIO_OUT_DRIVE_6;
	gpio.flt   = GPIO_FILTER_DISABLE;
	gpio.type  = GPIO_TYPE_CMOS;
	gpio.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_10, &gpio);

	/* Initialize rx pin */
	gpio.mode  = GPIO_MODE_INPUT;
	gpio.odos  = GPIO_PUSH_PULL;
	gpio.pupd  = GPIO_PUSH_UP;
	gpio.podrv = GPIO_OUT_DRIVE_6;
	gpio.nodrv = GPIO_OUT_DRIVE_6;
	gpio.flt   = GPIO_FILTER_DISABLE;
	gpio.type  = GPIO_TYPE_CMOS;
	gpio.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_11, &gpio);

	/* Initialize uart */
	s_h_uart.perh             = UART0;
	s_h_uart.init.baud        = 115200;
	s_h_uart.init.word_length = UART_WORD_LENGTH_8B;
	s_h_uart.init.stop_bits   = UART_STOP_BITS_1;
	s_h_uart.init.parity      = UART_PARITY_NONE;
	s_h_uart.init.mode        = UART_MODE_UART;
	s_h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	s_h_uart.tx_cplt_cbk      = NULL;
	s_h_uart.rx_cplt_cbk      = NULL;
	s_h_uart.error_cbk        = NULL;
	ald_uart_init(&s_h_uart);

	return;
}
/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    static uint8_t s_uart_buf[64];
    va_list args;

    va_start(args, fmt);
    vsnprintf((char*)s_uart_buf, 64, fmt, args);
    va_end(args);

    ald_uart_send(&s_h_uart, s_uart_buf, strlen((char*)s_uart_buf), 1000);
    ald_delay_ms(1);
    return;
}
/**
  * @}
  */
/**
  * @}
  */
