/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Jan 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */
static void uart_output(const char *str);

/* Private Variables --------------------------------------------------------- */
static uart_handle_t s_h_uart;

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief Jump tp app code
  * @retval None (nerver return)
  */
void jump2app(void)
{
    uint32_t app_addr;
    
    app_addr = *(uint32_t *)(APP_FLASH_S + 4);
    
    if (app_addr==0xFFFFFFFF)
    {
        uart_output("[boot] No app found!\r\n");
        return;
    }
    ald_uart_reset(&s_h_uart);
    ald_vtor_config(APP_FLASH_S, ENABLE);
    __set_MSP(*(uint32_t *)APP_FLASH_S);
    ((void(*)(void))(app_addr))();
}

/**
  * @brief  Clear flag in info page
  * @retval 0-success, other value indicates failed.
  */
void clear_flag_and_info(void)
{
    ald_iap_erase_page(INFO_FLAG_FLASH_S);
}

/**
  * @brief  Check app CRC
  * @param  page: total page of flash data.
  * @retval 0-success, other value indicates failed.
  */
uint8_t check_app_crc(void)
{
    uint32_t begin, end;
    crc_handle_t h_crc;
    uint32_t applen;
    uint32_t appcrc;
    uint32_t chkcrc;

    appcrc = *((uint32_t *)(UPDATE_INFO_CRC_VALUE));
    applen = *((uint32_t *)(UPDATE_INFO_CRC_LEN));
    begin = UPDATE_FLASH_S;
    end = UPDATE_FLASH_S + applen;

    /* Update address/length error */
    if (applen > APP_FLASH_SIZE || (begin & 0x3) || (end & 0x3))
        return 2;

    /* Initialize CRC */
    h_crc.perh          = CRC;
    h_crc.init.mode     = CRC_MODE_32;
    h_crc.init.seed     = 0xFFFFFFFF;
    h_crc.init.data_rev = ENABLE;
    h_crc.init.data_inv = DISABLE;
    h_crc.init.chs_rev  = ENABLE;
    h_crc.init.chs_inv  = ENABLE;
    ald_crc_init(&h_crc);
    chkcrc=ald_crc_calculate(&h_crc, (uint8_t *)begin, applen);
    ald_crc_reset(&h_crc);
    
    /* CRC error */
    if (chkcrc != appcrc)
        return 3;
    
    return 0;
}

/**
  * @brief  Check crc and update app
  * @retval 0-success(no update or app updated successfully) 1-update failed
  */
int update_app(void)
{
    static uint8_t s_buf[FLASH_PAGE_SIZE];
    uint32_t len = *((uint32_t *)UPDATE_INFO_CRC_LEN);
    uint32_t ptr = 0U;
    int result = 0;

    if (len == 0xFFFFFFFF)
    {
        return 0;
    }
    
    uart_output("[boot] Updating app\r\n");

    if (check_app_crc())
    {
        uart_output("[boot] CRC check failed\r\n");
        clear_flag_and_info();
        return 0;
    }
        
    for (; ptr < len; ptr += FLASH_PAGE_SIZE)
    {
        uint32_t dlen=FLASH_PAGE_SIZE;
        if (ptr+FLASH_PAGE_SIZE>len)
        {
            dlen=len-ptr;
            memcpy(s_buf,(uint8_t*)(UPDATE_FLASH_S+ptr),dlen);
            memset(s_buf+dlen,0xFF,FLASH_PAGE_SIZE-dlen);
        }
        else
        {
            memcpy(s_buf,(uint8_t*)(UPDATE_FLASH_S+ptr),FLASH_PAGE_SIZE);
        }
        if (ald_iap_program_words(APP_FLASH_S + ptr, s_buf, dlen, 1))
        {
            uart_output("[boot] Write failed\r\n");
            result = 3;
            break;
        }

        for (uint32_t i = ptr; i < ptr + (FLASH_PAGE_SIZE / 4); i += 4)
        {
            if (*(uint32_t *)(APP_FLASH_S + i) != *(uint32_t *)(UPDATE_FLASH_S + i))
            {
                /* Check failed */
                uart_output("[boot] Update failed\r\n");
                result = 3;
                goto UPDATE_END;
            }
        }
    }

UPDATE_END:
    if (result==0)
    {
        uart_output("[boot] App update success\r\n");
    }
    clear_flag_and_info();
    return result;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart0_init(void)
{
    gpio_init_t gpio;

    /* Initialize tx pin */
    gpio.mode  = GPIO_MODE_OUTPUT;
    gpio.odos  = GPIO_PUSH_PULL;
    gpio.pupd  = GPIO_PUSH_UP;
    gpio.podrv = GPIO_OUT_DRIVE_6;
    gpio.nodrv = GPIO_OUT_DRIVE_6;
    gpio.flt   = GPIO_FILTER_DISABLE;
    gpio.type  = GPIO_TYPE_CMOS;
    gpio.func  = GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_10, &gpio);

    /* Initialize rx pin */
    gpio.mode  = GPIO_MODE_INPUT;
    gpio.odos  = GPIO_PUSH_PULL;
    gpio.pupd  = GPIO_PUSH_UP;
    gpio.podrv = GPIO_OUT_DRIVE_6;
    gpio.nodrv = GPIO_OUT_DRIVE_6;
    gpio.flt   = GPIO_FILTER_DISABLE;
    gpio.type  = GPIO_TYPE_CMOS;
    gpio.func  = GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_11, &gpio);

    /* Initialize uart */
    s_h_uart.perh             = UART0;
    s_h_uart.init.baud        = 115200;
    s_h_uart.init.word_length = UART_WORD_LENGTH_8B;
    s_h_uart.init.stop_bits   = UART_STOP_BITS_1;
    s_h_uart.init.parity      = UART_PARITY_NONE;
    s_h_uart.init.mode        = UART_MODE_UART;
    s_h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    s_h_uart.tx_cplt_cbk      = NULL;
    s_h_uart.rx_cplt_cbk      = NULL;
    s_h_uart.error_cbk        = NULL;
    ald_uart_init(&s_h_uart);

    return;
}

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void uart_output(const char *str)
{
    ald_uart_send(&s_h_uart, (uint8_t *)str, strlen(str), 1000);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

    /* Enable peripheral clock */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    uart0_init();

    if (!update_app())
    {
        /* Update success or no update, try to jump to app */
        jump2app();
    }
    
    /* If no app found */
    while (1);
}

/**
  * @}
  */
/**
  * @}
  */
