#include <cstdlib>
#include <cstring>
#include <stdio.h>
#include <stdint.h>
/* Quantizer step size lookup table */
const uint16_t StepSizeTable[89]={7,8,9,10,11,12,13,14,16,17,
                            19,21,23,25,28,31,34,37,41,45,
                            50,55,60,66,73,80,88,97,107,118,
                            130,143,157,173,190,209,230,253,279,307,
                            337,371,408,449,494,544,598,658,724,796,
                            876,963,1060,1166,1282,1411,1552,1707,1878,2066,
                            2272,2499,2749,3024,3327,3660,4026,4428,4871,5358,
                            5894,6484,7132,7845,8630,9493,10442,11487,12635,13899,
                            15289,16818,18500,20350,22385,24623,27086,29794,32767};
/* Table of index changes */
const int8_t IndexTable[16]={-1,-1,-1,-1,2,4,6,8,-1,-1,-1,-1,2,4,6,8};

/**
  * @brief  ADPCM_Encode.
  * @param sample: a 16-bit PCM sample
  * @retval : a 4-bit ADPCM sample
  */
uint8_t ADPCM_Encode(int32_t sample)
{
  static int16_t  index = 0;
  static int32_t predsample = 0;
  uint8_t code=0;
  uint16_t tmpstep=0;
  int32_t diff=0;
  int32_t diffq=0;
  uint16_t step=0;
  
  step = StepSizeTable[index];

  /* 2. compute diff and record sign and absolut value */
  diff = sample-predsample;
  if (diff < 0)  
  {
    code=8;
    diff = -diff;
  }    
  
  /* 3. quantize the diff into ADPCM code */
  /* 4. inverse quantize the code into a predicted diff */
  tmpstep = step;
  diffq = (step >> 3);

  if (diff >= tmpstep)
  {
    code |= 0x04;
    diff -= tmpstep;
    diffq += step;
  }
  
  tmpstep = tmpstep >> 1;

  if (diff >= tmpstep)
  {
    code |= 0x02;
    diff -= tmpstep;
    diffq+=(step >> 1);
  }
  
  tmpstep = tmpstep >> 1;
  
  if (diff >= tmpstep)
  {
    code |=0x01;
    diffq+=(step >> 2);
  }
  
  /* 5. fixed predictor to get new predicted sample*/
  if (code & 8)
  {
    predsample -= diffq;
  }
  else
  {
    predsample += diffq;
  }  

  /* check for overflow*/
  if (predsample > 32767)
  {
    predsample = 32767;
  }
  else if (predsample < -32768)
  {
    predsample = -32768;
  }
  
  /* 6. find new stepsize index */
  index += IndexTable[code];
  /* check for overflow*/
  if (index <0)
  {
    index = 0;
  }
  else if (index > 88)
  {
    index = 88;
  }
  
  /* 8. return new ADPCM code*/
  return (code & 0x0f);
}



/**
  * @brief  ADPCM_Decode.
  * @param code: a byte containing a 4-bit ADPCM sample. 
  * @retval : 16-bit ADPCM sample
  */
int16_t ADPCM_Decode(uint8_t code)
{
  static int16_t  index = 0;
  static int32_t predsample = 0;
  uint16_t step=0;
  int32_t diffq=0;
  
  step = StepSizeTable[index];

  /* 2. inverse code into diff */
  diffq = step>> 3;
  if (code&4)
  {
    diffq += step;
  }
  
  if (code&2)
  {
    diffq += step>>1;
  }
  
  if (code&1)
  {
    diffq += step>>2;
  }

  /* 3. add diff to predicted sample*/
  if (code&8)
  {
    predsample -= diffq;
  }
  else
  {
    predsample += diffq;
  }
  
  /* check for overflow*/
  if (predsample > 32767)
  {
    predsample = 32767;
  }
  else if (predsample < -32768)
  {
    predsample = -32768;
  }

  /* 4. find new quantizer step size */
  index += IndexTable [code];
  /* check for overflow*/
  if (index < 0)
  {
    index = 0;
  }
  if (index > 88)
  {
    index = 88;
  }
  
  /* 5. save predict sample and index for next iteration */
  /* done! static variables */
  
  /* 6. return new speech sample*/
  return ((int16_t)predsample);
}
typedef struct
{
    uint32_t   ChunkID;       /* 0 */
    uint32_t   FileSize;      /* 4 */
    uint32_t   FileFormat;    /* 8 */

    uint32_t   SubChunk1ID;   /* 12 */
    uint32_t   SubChunk1Size; /* 16*/
    uint16_t   AudioFormat;   /* 20 */
    uint16_t   NbrChannels;   /* 22 */
    uint32_t   SampleRate;    /* 24 */

    uint32_t   ByteRate;      /* 28 */
    uint16_t   BlockAlign;    /* 32 */
    uint16_t   BitPerSample;  /* 34 */
    uint32_t   SubChunk2ID;   /* 36 */
    uint32_t   SubChunk2Size; /* 40 */

} wave_formattypedef_t;
#define WAVF_RIFF 0x46464952
#define WAVF_WAVE 0x45564157
#define WAVF_fmt  0x20746D66
#define WAVF_data 0x61746164
#define U2C4(X) (X)&0xFF,(X>>8)&0xFF,(X>>16)&0xFF,(X>>24)&0xFF

void print_wav_info(wave_formattypedef_t *wav)
{
    printf("Chunk ID     = %X,%c%c%c%c\r\n", wav->ChunkID, U2C4(wav->ChunkID));
    printf("FileSize     = %d\r\n", wav->FileSize);
    printf("FileFormat   = %d,%c%c%c%c\r\n", wav->FileFormat, U2C4(wav->FileFormat));
    printf("SubChunk1ID  = 0x%08X,%c%c%c%c\r\n", wav->SubChunk1ID, U2C4(wav->SubChunk1ID));
    printf("SubChunk1Size= %d\r\n", wav->SubChunk1Size);
    printf("AudioFormat  = %d\r\n", wav->AudioFormat);
    printf("NbrChannels  = %d\r\n", wav->NbrChannels);
    printf("SampleRate   = %d\r\n", wav->SampleRate);
    printf("ByteRate     = %d\r\n", wav->ByteRate);
    printf("BlockAlign   = %d\r\n", wav->BlockAlign);
    printf("BitPerSample = %d\r\n", wav->BitPerSample);
    printf("SubChunk2ID  = 0x%08X,%c%c%c%c\r\n", wav->SubChunk2ID, U2C4(wav->SubChunk2ID));
    printf("SubChunk2Size= %d\r\n", wav->SubChunk2Size);
}
void cvt_wav(const char*infile,const char*outfile,bool is_encode)
{
    FILE*fp=fopen(infile,"rb");
    wave_formattypedef_t wav_header;
    int wav_header_size=44,wav_data_size=0;

    fread(&wav_header,sizeof(wave_formattypedef_t),1,fp);
    printf("Input file:%s\n",infile);
    printf("--SampleRate=%d\n",wav_header.SampleRate);
    printf("--BitPerSample=%d\n",wav_header.BitPerSample);
    printf("--NbrChannels=%d\n",wav_header.NbrChannels);

   // print_wav_info(&wav_header);
    if (wav_header.SubChunk2ID==WAVF_data)
    {
        wav_data_size=wav_header.SubChunk2Size;
    }
    else
    {
        /* This is not data find next subchunk*/
        int subchunk=3;
        wav_header_size+=wav_header.SubChunk2Size+8;
        while (1)
        {
            uint32_t t[2];
            fseek(fp, wav_header_size-8, SEEK_SET); 
            fread(t,8,1,fp);
            //printf("SubChunk%dID  = 0x%08X,%c%c%c%c\r\n",subchunk,t[0],U2C4(t[0]));
            //printf("SubChunk%dSize= %d\r\n",subchunk, t[1]);
            if (t[0]==WAVF_data)
            {
                wav_data_size=t[1];
                break;
            }
            else
            {
                wav_header_size+=t[1];
            }
        }
    }
    fseek(fp, wav_header_size, SEEK_SET); 
    printf("--DataSize=%d\n",wav_data_size);
    if (is_encode)
    {
        if (wav_header.BitPerSample!=16)
        {
            printf("Not support bits=%d (Only support encode 16bit wav)\n",wav_header.BitPerSample);
            return;
        }
        uint32_t alloc_size=(wav_data_size+3)&(~0x3);
        int16_t*inbuf=(int16_t*)malloc(alloc_size);
        uint8_t*outbuf=(uint8_t*)malloc(alloc_size/4); //16bit -> 4bit
        fread(inbuf,wav_data_size,1,fp);
        if (alloc_size!=wav_data_size){
            memset((uint8_t*)inbuf+wav_data_size,0,alloc_size-wav_data_size);
        }
        for (int i=0;i<wav_data_size/2;i+=2)
        {
            uint8_t data=0;
            outbuf[i/2]=ADPCM_Encode(inbuf[i]);
            outbuf[i/2]|=ADPCM_Encode(inbuf[i+1])<<4;
        }

        wave_formattypedef_t wh;
        wh.ChunkID=WAVF_RIFF;
        wh.FileSize=44+alloc_size/4;
        wh.FileFormat=WAVF_WAVE;
        wh.SubChunk1ID=WAVF_fmt;
        wh.SubChunk1Size=16;
        wh.AudioFormat=1;
        wh.NbrChannels=1;
        wh.SampleRate=wav_header.SampleRate;
        wh.ByteRate=wav_header.SampleRate/2;
        wh.BitPerSample=4;
        wh.SubChunk2ID=WAVF_data;
        wh.SubChunk2Size=alloc_size/4;

        FILE*fw=fopen(outfile,"wb");
        fwrite(&wh,44,1,fw);
        fwrite(outbuf, alloc_size/4, 1, fw);
        fclose(fw);

        printf("Encoded OK\nSave to %s\n",outfile);
        printf("--File size=%d\n",44+alloc_size/4);
    }
    else{
        if (wav_header.BitPerSample!=4)
        {
            printf("Not support bits=%d (Only support decode 4bit wav)\n",wav_header.BitPerSample);
            return;
        }
        uint32_t alloc_size=wav_data_size*4;
        uint8_t*inbuf=(uint8_t*)malloc(wav_data_size);
        uint16_t*outbuf=(uint16_t*)malloc(wav_data_size*4); //4bit -> 16bit
        fread(inbuf,wav_data_size,1,fp);

        for (int i=0;i<wav_data_size;i++)
        {
            uint8_t data=0;
            outbuf[i*2]=ADPCM_Decode((inbuf[i])&0xF);
            outbuf[i*2+1]=ADPCM_Decode((inbuf[i]>>4)&0xF);
        }

        wave_formattypedef_t wh;
        wh.ChunkID=WAVF_RIFF;
        wh.FileSize=44+wav_data_size*4;
        wh.FileFormat=WAVF_WAVE;
        wh.SubChunk1ID=WAVF_fmt;
        wh.SubChunk1Size=16;
        wh.AudioFormat=1;
        wh.NbrChannels=1;
        wh.SampleRate=wav_header.SampleRate;
        wh.ByteRate=wav_header.SampleRate*2;
        wh.BitPerSample=16;
        wh.SubChunk2ID=WAVF_data;
        wh.SubChunk2Size=wav_data_size*4;
        FILE*fw=fopen(outfile,"wb");
        fwrite(&wh,44,1,fw);
        fwrite(outbuf, wav_data_size*4, 1, fw);
        fclose(fw);
        printf("Decoded OK\nSave to %s\n",outfile);
        printf("--File size=%d\n",44+wav_data_size*4);
    }
    fclose(fp);
}

void show_usage()
{
    printf("adpcm [-e|-d] infile outfile\n");
    printf("Encode/Decode adpcm in wav format\n");
    printf("\tArgs:\n");
    printf("\t-e        encode mode\n");
    printf("\t-d        decode mode\n");
    printf("\tinfile    input wav file\n");
    printf("\toutfile   output wav file\n");
}
int main(int argc,char**argv)
{
    #ifdef __DEBUG
    cvt_wav("test.wav","test1.wav",true);
    cvt_wav("test1.wav","test2.wav",false);
    #else
    bool is_encode=true;
    for (int i=1;i<argc;i++)
    {
        if (argv[i][0]=='-')
        {
            if (argv[i][1]=='e')
            {
                is_encode=true;
            }
            else if (argv[i][1]=='d')
            {
                is_encode=false;
            }
            else if (argv[i][1]=='h')
            {
                show_usage();
            }
        }
        else
        {
            if (i+1<argc)
            {
                cvt_wav(argv[i],argv[i+1],is_encode);
            }
        }
    }
    #endif
    return 0;
}