/**********************************************************************************
 *
 * @file    xmodem_frame.c
 * @brief   xmodem send/recv
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include <string.h>
#include <stdio.h>
#include "xmodem_frame.h"
#include "stdio_uart.h"

/** @addtogroup Bootloader
  * @{
  */
/** @defgroup Frame Frame
  * @brief    Bootloader Frame Module
  * @{
  */
/** @defgroup Frame_Public_Variables Public Variable
  * @brief    Frame Public Variable
  * @{
  */
boot_frame_env_t frame_env;
/**
  * @}
  */

/** @defgroup Frame_Private_Functions Private Functions
  * @brief    Frame Private Functions
  * @{
  */
/**
  * @brief  Send command to tramsmitor
  * @param  cmd: Command.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_cmd_send(xmod_tx_cmd_t cmd)
{
    uint8_t _cmd = cmd;

    if ((uart_send_str((char *)&_cmd, 1, 100)))
    {
        frame_env.state = UART_SEND_ERROR;
        return -1;
    }

    return 0;
}

/**
  * @brief  Check frame index.
  * @param  idx: frame index.
  * @retval 0-success, other value indicates failed.
  */
static int check_idx(uint8_t idx)
{
    uint8_t _idx  = 0;
    uint8_t next_idx = 0;

    _idx = frame_env.idx;
    next_idx = _idx == 0xFF ? 0 : _idx + 1;

    if ((next_idx != idx) && (idx != _idx))
        return 1;

    return 0;
}

#ifdef XMODE_1K
/**
  * @brief  Receive data by 1K_XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_1K_recv(void)
{
    uint16_t crc, _crc;

    if ((uart_recv_str((char *)&frame_env.soh, 1, TIME_OUT)))
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if (frame_env.soh == XMODE_EOT)
    {
        frame_env.state = FRAME_SUCCESS;
        return 1;
    }

    if (frame_env.soh != XMODE_STX)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }
    if ((uart_recv_str((char *)frame_env.data, 1028, TIME_OUT)))
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if ((frame_env.data[0] ^ frame_env.data[1]) != 0xFF)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    if ((check_idx(frame_env.data[0])) != 0)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    _crc = calc_crc_ccitt(&frame_env.data[2], 0, 1024);
    crc = (frame_env.data[1026] << 8) | frame_env.data[1027];

    if (crc != _crc)
    {
        frame_env.state = FRAME_VERIFY_ERROR;
        return -3;
    }

    if (frame_env.data_cbk(frame_env.usr_data, frame_env.idx, &frame_env.data[2], crc))
    {
        frame_env.state = DATA_CBK_ERROR;
        return -4;
    }

    ++frame_env.idx;
    return 0;
}


/**
  * @brief  Receive data by 1K_XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_1K_send(void)
{
    uint16_t crc;
    int ret = 0;
    int retry = 3;
    int len = 0; 
    
    len = frame_env.data_cbk(frame_env.usr_data, frame_env.idx, &frame_env.data[3], 0);
    if (len == 0)
    {
        frame_env.data[0] = XMODE_EOT;
        uart_send_str((char *)&frame_env.data[0], 1, TIME_OUT);
        return 1;
    }
    if (len != 1024)
    {
        memset(&frame_env.data[2 + len], XMODE_NACK, 1024 - len);
    }
    frame_env.data[0] = XMODE_STX;
    frame_env.data[1] = frame_env.idx;
    frame_env.data[2] = ~frame_env.idx;
    crc = calc_crc_ccitt(&frame_env.data[3], 0, 1024);
    frame_env.data[1027] = crc >> 8;
    frame_env.data[1028] = crc & 0xFF;
    
    retry = 10;
    while (retry-- > 0)
    {
        uint8_t ackdata;
        uart_reset_rx_fifo();

        if ((uart_send_str((char *)frame_env.data, 1029, TIME_OUT)))
        {
            frame_env.state = FRAME_SEND_TIMEOUT;
            ret = -1;
            continue;
        }
        
        if (uart_recv_str((char*)&ackdata,1,TIME_OUT))
        {
            frame_env.state = FRAME_RECEIVE_TIMEOUT;
            ret = -1;
            continue;
        }
        
        if (ackdata != XMODE_ACK)
        {
            frame_env.state = FRAME_SEND_NACK;
            ret = -3;
            continue;
        }
        ++frame_env.idx;
        return 0;
    }
    return ret;
}
#endif

#ifdef XMODE_128
/**
  * @brief  Receive data by XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_128_recv(void)
{
    uint16_t crc, _crc;
    uint8_t xmode_buf[132];
    uint8_t tmp = 0;
    uint16_t page = 0;

    memset(xmode_buf, 0, sizeof(xmode_buf));

    if ((uart_recv_str((char *)&frame_env.soh, 1, TIME_OUT)))
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if (frame_env.soh == XMODE_EOT)
    {
        frame_env.state = FRAME_SUCCESS;

        if (frame_env.idx % 8 == 0)
            return 1;

        frame_env.state = FRAME_INDEX_ERROR;
        /* APP size must be multiple of 1024 */
        /*
        MD_CRC_RESET();
        for (int i=0;i<1024;i++)
        {
            md_crc_write_data(h_crc, frame_env.data[i]);
        }
        crc = md_crc_get_check_result(h_crc);

        page = frame_env.idx / 8;

        if (write_data_to_flash(page, frame_env.data, crc))
        {
            memset(frame_env.data, 0, sizeof(frame_env.data));
            frame_env.state = FLASH_WRITE_ERROR;
            return -4;
        }
        */
        return -1;
    }

    if ((uart_recv_str((char *)xmode_buf, 132, TIME_OUT)))
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if ((xmode_buf[0] ^ xmode_buf[1]) != 0xFF)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    if ((check_idx(xmode_buf[0])) != 0)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    MD_CRC_RESET();
    for (int i = 0; i < 128; i++)
    {
        md_crc_write_data(h_crc, xmode_buf[2 + i]);
    }
    _crc = md_crc_get_check_result(h_crc);
    crc = (xmode_buf[130] << 8) | xmode_buf[131];

    if (crc != _crc)
    {
        frame_env.state = FRAME_VERIFY_ERROR;
        return -3;
    }

    tmp = (frame_env.idx % 8);
    memcpy((char *)&frame_env.data[tmp * 128], (const char *)&xmode_buf[2], 128);

    if (tmp == 7)
    {
        MD_CRC_RESET();
        for (int i = 0; i < 1024; i++)
        {
            md_crc_write_data(h_crc, frame_env.data[i]);
        }
        crc = md_crc_get_check_result(h_crc);

        page = frame_env.idx / 8;

        if (write_data_to_flash(page, frame_env.data, crc))
        {
            memset(frame_env.data, 0, sizeof(frame_env.data));
            frame_env.state = FLASH_WRITE_ERROR;
            return -4;
        }

        memset(frame_env.data, 0, sizeof(frame_env.data));
    }

    ++frame_env.idx;
    return 0;
}
#endif

/**
  * @brief  Receive a frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_frame_recv(void)
{
    int result = 0;
#ifdef XMODE_128
    result = xmod_128_recv();
#endif

#ifdef XMODE_1K
    result = xmod_1K_recv();
#endif
    return result;
}

/**
  * @}
  */

/** @defgroup Frame_Public_Functions Public Functions
  * @brief    Frame Public Functions
  * @{
  */

/**
  * @brief  Show update info
  * @retval none.
  */
const char *xm_get_info(void)
{
    switch (frame_env.state)
    {
        case FRAME_SUCCESS         :   /**< Frame transmission sucess */
            return "FRAME_SUCCESS";
        case FRAME_INDEX_ERROR     :   /**< Frame index error */
            return "FRAME_INDEX_ERROR";
        case FRAME_RECEIVE_TIMEOUT :   /**< Timeout error */
            return "FRAME_RECEIVE_TIMEOUT";
        case FRAME_VERIFY_ERROR    :   /**< Frame verigy error */
            return "FRAME_VERIFY_ERROR";
        case UART_SEND_ERROR       :   /**< UART send error */
            return "UART_SEND_ERROR";
        case DATA_CBK_ERROR        :   /**< Data callback returns an error */
            return "DATA_CBK_ERROR";
        default:
            return "UNKNOWN_ERROR";
    }
}

/**
  * @brief  Initialize frame module
  * @param  hperh: Pointer to uart_handle_t structure.
  * @retval None
  */
void xm_frame_init(XM_FRAME_DATA_CBK data_cbk, void *usrdata)
{
    memset(&frame_env, 0, sizeof(boot_frame_env_t));
    frame_env.data_cbk = data_cbk;
    frame_env.usr_data = usrdata;
    return;
}

/**
  * @brief  download bootloader program.
  * @retval 0-success, other value indicates failed.
  */
int  xm_download(void)
{
    int ret;

    frame_env.idx   = 0;
    frame_env.cnt   = 0;
    frame_env.state = FRAME_SUCCESS;
    xmod_cmd_send(XMOD_TX_CMD_START);

    while (1)
    {
        ret = xmod_frame_recv();

        if ((ret < 0) && (++frame_env.cnt < 10) && frame_env.state == FRAME_RECEIVE_TIMEOUT)
        {
            if (frame_env.idx == 0)
                xmod_cmd_send(XMOD_TX_CMD_START);
            else
                xmod_cmd_send(XMOD_TX_CMD_NCK);

            continue;
        }

        if (ret < 0)
        {
            xmod_cmd_send(XMOD_TX_CMD_TERM);
            return ret;
        }

        if (ret == 1)
        {
            xmod_cmd_send(XMOD_TX_CMD_ACK);
            return 0;
        }

        xmod_cmd_send(XMOD_TX_CMD_ACK);
    }
}


/**
  * @brief  download bootloader program.
  * @retval 0-success, other value indicates failed.
  */
int  xm_send(void)
{
    int ret;
    int retry=5;
    frame_env.idx   = 1;
    frame_env.cnt   = 0;
    frame_env.state = FRAME_SUCCESS;

    while (retry-->0)
    {
        uint8_t ackdata = 0;
        if (uart_recv_str((char *)&ackdata, 1, TIME_OUT))
        {
            frame_env.state = FRAME_RECEIVE_TIMEOUT;
            ret=-1;
            continue;
        }
        if (ackdata == XMODE_CRC)
        {
            break;
        }
        else
        {
            frame_env.state = FRAME_INDEX_ERROR;
            ret = -2;
        }
    }
    if (retry<=0)
    {
        return ret;
    }
    while (1)
    {
        ret = xmod_1K_send();

        if (ret < 0)
        {
            xmod_cmd_send(XMOD_TX_CMD_TERM);
            return ret;
        }

        if (ret == 1)
        {
            break;
        }
    }
    
    while (1)
    {
        uint8_t ackdata = 0;
        if (uart_recv_str((char *)&ackdata, 1, TIME_OUT))
        {
            break;
        }
        if (ackdata == XMODE_NACK)
        {
            ackdata=XMODE_EOT;
            uart_send_str((char *)&ackdata, 1, TIME_OUT);
            continue;
        }
        else if (ackdata == XMODE_NACK)
        {
            break;
        }
    }
    return 0;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
