/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *          20 Mar 2021     shiwa           add chip erase and wait for specific status
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

static spi_handle_t s_spi;
flash_op_cbk flash_write_cplt=NULL;
flash_op_cbk flash_read_cplt=NULL;
/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

static void init_spi_pin(void);
static void flash_spi_rx_cplt(struct spi_handle_s *arg);
ald_status_t flash_wait_status(uint32_t mask,uint32_t expect);
/* Private Function ---------------------------------------------------------- */


/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval Status.
  */
ald_status_t flash_sector_erase(uint32_t addr)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;       /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;
    
    /* Wait if it is doing something */
    flash_wait_unbusy();
    
    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != OK)  /*First write send enabling instruction*/
    {
        FLASH_CS_SET();
        return ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    flash_wait_status(0x03,0x02);
    
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }

    FLASH_CS_SET();

    return OK;
}

/**
  * @brief  Erase full chip
  * @retval Status.
  */
ald_status_t flash_chip_erase()
{
    
    /* Wait if it is doing something */
    flash_wait_unbusy();
    
    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != OK)  /*First write send enabling instruction*/
    {
        FLASH_CS_SET();
        return ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    flash_wait_status(0x03,0x02);
    
    FLASH_CS_CLR();

    if (ald_spi_send_byte_fast(&s_spi, FLASH_CHIP_ERASE) != OK) 
    {
        FLASH_CS_SET();
        return ERROR;
    }

    FLASH_CS_SET();

    return OK;
}
/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_write(uint32_t addr, void *data, uint32_t size)
{
    uint8_t cmd_buf[4];
    uint32_t i = 0U;
    uint8_t*buf=(uint8_t*)data;

    if (buf == NULL)
        return ERROR;
    
    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;
    
    /* Wait if it is doing something */
    flash_wait_unbusy();
    
    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != OK)
    {
        FLASH_CS_SET();
        return ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    flash_wait_status(0x03,0x02);
    
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        if (ald_spi_send_byte_fast(&s_spi, buf[i]) != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }

    FLASH_CS_SET();

    return OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_read(uint32_t addr, void *data, uint32_t size)
{
    uint8_t cmd_buf[4];
    uint32_t i = 0U;
    uint8_t*buf=data;
    int r_flag = 0;

    if (buf == NULL)
        return ERROR;
    
    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;
    
    /* Wait if it is doing something */
    flash_wait_unbusy();
    
    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*Send the programming instructions and Flash address three bytes*/
    {
        buf[i] = ald_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }

    FLASH_CS_SET();

    return OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_read_dma(uint32_t addr, void *data, uint32_t size)
{
    uint8_t cmd_buf[4];
    uint32_t i = 0U;
    uint8_t*buf=data;

    if (buf == NULL)
        return ERROR;
    
    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;
    
    /* Wait if it is doing something */
    flash_wait_unbusy();
    
    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return ERROR;
        }
    }

    s_spi.rx_cplt_cbk=flash_spi_rx_cplt;
    ald_spi_recv_by_dma(&s_spi,buf,size,3);

    return OK;
}


ald_status_t flash_wait_status(uint32_t mask,uint32_t expect)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != OK)   /*Send to read status command*/
    {
        FLASH_CS_SET();     /*Pick up and release of Flash*/
        return ERROR;
    }

    do
    {
        status = ald_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }
    while ((status & mask)!=expect);

    FLASH_CS_SET();
    return OK;
}
/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_wait_unbusy(void)
{
    flash_wait_status(0x01,0x00);

    return OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (ald_spi_send_byte_fast(&s_spi, flash_id[i]) != OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = ald_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

static void flash_spi_rx_cplt(struct spi_handle_s *arg)
{
    FLASH_CS_SET();
    
    if (flash_read_cplt)
    {
        flash_read_cplt();
    }
}
/**
  * @brief spi function
  * @retval None.
  */
void init_mcu_spi(void)
{
    init_spi_pin();

    s_spi.perh           = SPI0;               /*Using SPI0*/
    s_spi.init.mode      = SPI_MODE_MASTER;    /*SPI host mode*/
    s_spi.init.baud      = SPI_BAUD_8;         /*clock / 8*/
    s_spi.init.data_size = SPI_DATA_SIZE_8;    /*8 bit pattern*/
    s_spi.init.polarity  = SPI_CPOL_HIGH;      /*Free high level*/
    s_spi.init.phase     = SPI_CPHA_SECOND;    /*The second edge receiving data*/
    s_spi.init.first_bit = SPI_FIRSTBIT_MSB;   /*Send the MSB first*/
    s_spi.init.dir       = SPI_DIRECTION_2LINES;
    s_spi.init.ss_en     = DISABLE;
    s_spi.init.crc_calc  = DISABLE;

    ald_spi_init(&s_spi);  /*According to the parameter initialization SPI peripherals*/
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void init_spi_pin(void)
{
    gpio_init_t gpio_init;

    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.nodrv = GPIO_OUT_DRIVE_1;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_1;

    ald_gpio_init(SPI_NSS_PORT, SPI_NSS_PIN, &gpio_init);  /*Initialize the selected pin*/
    FLASH_CS_SET();   /*Choose the output high, release of Flash*/

    gpio_init.func = GPIO_FUNC_4;
    ald_gpio_init(SPI_SCK_PORT, SPI_SCK_PIN, &gpio_init);  /*Initialize the clock output pin*/

    gpio_init.func = GPIO_FUNC_4;
    ald_gpio_init(SPI_MOSI_PORT, SPI_MOSI_PIN, &gpio_init);  /*Initialize the MOSI pin*/

    gpio_init.mode = GPIO_MODE_INPUT;
    gpio_init.func = GPIO_FUNC_4;
    ald_gpio_init(SPI_MISO_PORT, SPI_MISO_PIN, &gpio_init);  /*Initializes the MISO pins*/
}

/**
  * @brief  delay some time.
  * @retval None.
  */
void internal_delay(int i)
{
    while (i--) ;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
