/**********************************************************************************
 *
 * @file    shell_util.c
 * @brief   a simple shell
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <stdio.h>
#include <string.h>
#include "shell_util.h"
#include "stdio_uart.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

int split_cmd(char *s, char **args, int len)
{
    int p = 0;
    char last = ' ';
    while (*s && p < len)
    {
        if (last == ' ' && *s != ' ')
        {
            args[p] = s;
            p++;
        }
        last = *s;
        if (*s == ' ')
            *s = 0;
        s++;
    }
    return p;
}

uint32_t get_uint(const char *s)
{
    uint32_t res = 0;
    int len = strlen(s);
    if (s[0] == 0)
        return 0;

    if (s[1] != 0 && s[0] == '0' && (s[1] == 'x' || s[1] == 'X'))
    {
        s += 2;
        while (*s)
        {
            int ch = (*s) ;
            int ch2 = (*s) & (~0x20);
            if (ch >= '0' && ch <= '9')
            {
                res = res * 16 + (ch - '0');
            }
            else if (ch2 >= 'A' && ch2 <= 'F')
            {
                res = res * 16 + (ch2 - 'A') + 10;
            }
            else
            {
                return res;
            }
            s++;
        }
        return res;
    }
    while (*s)
    {
        int ch = (*s);
        if (ch >= '0' && ch <= '9')
        {
            res = res * 10 + (ch - '0');
        }
        else
        {
            return res;
        }
        s++;
    }
    return res;
}

void do_shell_cmd(struct ShellEnv *env)
{
    char *args[8];
    int argc = 0;
    printf(">%s\r\n", env->buf);
    argc = split_cmd(env->buf, args, 8);
    struct CMD_FUNC *cmd = env->cmd_table;
    for (; cmd->func; cmd++)
    {
        if (!strcmp(args[0], cmd->name))
        {
            cmd->func(argc, args);
            break;
        }
    }
    if (cmd->func == NULL)
    {
        printf("Unknown command \"%s\"\r\n", args[0]);
    }
}

void shell_input(char ch, struct ShellEnv *env)
{
    if (env->buf_p > env->buf_len - 1)
    {
        env->buf_p = 0;
        return;
    }

    if (ch == '\0')
        return;

    env->buf[env->buf_p] = ch;
    env->buf_p++;

    if (env->buf_p >= 2)
    {
        if (env->buf[env->buf_p - 1] == '\n' && env->buf[env->buf_p - 2] == '\r')
        {
            env->buf[env->buf_p - 2] = 0;
            env->buf_p = env->buf_p - 2;
            do_shell_cmd(env);
            env->buf_p = 0;
        }
    }

}
void enter_shell_loop(struct ShellEnv *env)
{
    uart_enable_rx_irq(1);
    while ((env->flag & SHELL_EXIT) == 0)
    {
        char ch = uart_peak_ch();
        if (ch)
        {
            shell_input(ch, env);
        }
    }
    uart_enable_rx_irq(0);
}
