/**********************************************************************************
 *
 * @file    flash_util.c
 * @brief   a simple file struct for flash
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
 
/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "flash_util.h"
#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

 
/**
  * @brief List files in flash
  * @retval  0 ==> Success
  *         -1 ==> Read flash error
  *         -2 ==> End of files 
  *         -3 ==> Invalid index
  */
int flash_file_list(FlashFileIter*f)
{
    while (1)
    {
        //move to next index
        f->index++;
        if (f->index<0||f->index>64)
        {
            return -3;
        }
        if (f->index==64)
        {
            return -2;
        }
        
        uint32_t addr=0;
        addr=f->index*64;
        
        if (flash_read(addr,f,64))
        {
            return -1;
        }
        if (f->file.faddr==0||f->file.fsize==0)
        {
            continue;
        }
        if (f->file.faddr==0xFFFFFFFF||f->file.fsize==0xFFFFFFFF)
        {
            return -2;
        }
        f->file.name[56-1]=0;
        return 0;
    }
}
int flash_file_find(const char*name)
{
    FlashFileIter fi;
    fi.index=-1;
    while (flash_file_list(&fi)==0)
    {
        if (!strcmp(name,fi.file.name))
        {
            return fi.index;
        }
    }

    return -1;
}

/**
  * @brief Open file in flash
  * @retval  0 ==> Success
  *         -1 ==> Read flash error
  *         -2 ==> Index invalid
  *         -3 ==> File address or size invalid
  */
int flash_file_open(int index,FlashFile*file)
{
    uint8_t buf[64];
    uint32_t faddr;
    uint32_t fsize;
    
    if (index<0||index>=4096/64)
    {
        return -2;
    }
    if (flash_read(index*64, buf, 64))
    {
        return -1;
    }
    faddr = ((uint32_t *)buf)[0];
    fsize = ((uint32_t *)buf)[1];
    
    if (faddr==0xFFFFFFFF||fsize==0xFFFFFFFF)
    {
        return -3;
    }
    
    file->index=index;
    file->data_addr=faddr;
    file->data_size=fsize;
    file->read_ptr=0;
    file->flag=0;
    return 0;
}

/**
  * @brief Open file in flash
  * @retval >0 ==> Read bytes
  *          0 ==> No data left
  *         -1 ==> Read flash error
  */
int flash_file_read(FlashFile*file,void*buf,uint32_t len)
{
    int rlen=len;
    uint32_t read_addr;
    if (file->read_ptr+len>file->data_size)
    {
        rlen=file->data_size-file->read_ptr;
    }
    
    read_addr=file->read_ptr+file->data_addr;
    if (flash_read(read_addr,buf,rlen))
    {
        return -1;
    }
    
    file->read_ptr+=rlen;
    
    return rlen;
}

/**
  * @brief Open file in flash
  * @retval >0 ==> Read bytes
  *          0 ==> No data left
  *         -1 ==> Read flash error
  */
int flash_file_read_dma(FlashFile*file,void*buf,uint32_t len,flash_op_cbk cbk)
{
    int rlen=len;
    uint32_t read_addr;
    if (file->read_ptr+len>file->data_size)
    {
        rlen=file->data_size-file->read_ptr;
    }
    
    flash_read_cplt=cbk;
    
    read_addr=file->read_ptr+file->data_addr;
    
    if (flash_read_dma(read_addr,buf,rlen))
    {
        return -1;
    }
    
    file->read_ptr+=rlen;
    
    return rlen;
}


int flash_find_free_index(uint32_t *data_addr)
{
    uint32_t p = 0;
    uint8_t buf[64];
    uint32_t data_index = 0xFFFFFFFF;
    uint32_t data_ends = 4096;
    for (p = 0; p < 4096; p += 64)
    {
        flash_read(p, (char *)buf, 64);
        uint32_t faddr = ((uint32_t *)buf)[0];
        uint32_t fsize = ((uint32_t *)buf)[1];
        if (faddr == 0xFFFFFFFF)
        {
            if (data_index == 0xFFFFFFFF)
            {
                data_index = p;
            }
        }
        else
        {
            uint32_t fend = faddr + fsize;
            if (fend > data_ends)
            {
                data_ends = fend;
            }
        }
    }

    if (data_index != 0xFFFFFFFF)
    {
        *data_addr = (data_ends + 0x3FF) & (~0x3FF);
        return data_index / 64;
    }
    return -1;
}
/**
  * @brief Create a new file in flash
  * @param len: if len>0 then len will be write when open,else file size will be writen when close
  * @retval 
  *          0 ==> Success
  *         -1 ==> Cannot find a free index
  *         -2 ==> File name too long
  */
int flash_file_new(FlashFile*file,const char*name,uint32_t len)
{
    uint32_t data_addr;
    int idx=flash_find_free_index(&data_addr);
    uint32_t slen=strlen(name);
    if (idx<0)
    {
        return -1;
    }
    if (slen>=56)
    {
        return -2;
    }
    file->data_addr=data_addr;
    file->data_size=len;
    file->read_ptr=0;
    file->index=idx;
    file->flag=0;
    
    
    uint32_t flash_index_addr = idx * 64;
    flash_write(flash_index_addr,&file->data_addr,4);
    if (len>0)
    {
        flash_write(flash_index_addr+4,&file->data_size,4);
    }
    flash_write(flash_index_addr+OFFSETOF(name,struct FlashRecord),(void*)name,slen+1);
    return 0;
}

/**
  * @brief Close a file in flash, only necessary when use `flash_file_new` with len==0
  * @retval  0 ==> Success
  *         -1 ==> File already has a size
  *         -2 ==> Write fail
  */
int flash_file_close(FlashFile*file)
{
    uint32_t flash_index_addr = file->index * 64;
    if (file->data_size)
    {
        return -1;
    }
    file->data_size=file->read_ptr;
    if (flash_write(flash_index_addr+4,&file->data_size,4))
    {
        return -2;
    }
    return 0;
}
/**
  * @brief Write file in flash
  * @retval >0 ==> Writen bytes
  *         -1 ==> Erase error 
  *         -2 ==> Write error
  */
int flash_file_write(FlashFile*file,void*buf,uint32_t len)
{
    uint32_t wlen=0;
    while (wlen<len)
    {
        uint32_t waddr=file->data_addr+file->read_ptr;
        uint32_t wsize=256-(waddr&0xFF);
        if ((waddr&0x3FF)==0)
        {
            //a new sector, erase first 
            if (flash_sector_erase(waddr))
            {
                return -1;
            }
        }
        if (len<wlen+wsize)
        {
            wsize=len-wlen;
        }
        if (flash_write(waddr,(uint8_t*)buf+wlen,wsize))
        {
            return -2;
        }
        wlen+=wsize;
        file->read_ptr+=wsize;
    }
    
    return wlen;
}

/**
  * @brief  Erase pages for write
  *         `len` must be multiply of 1024 
  * @retval >0 ==> Writen bytes
  *         -1 ==> Erase error
  */
int flash_file_prepare_write(FlashFile*file,uint32_t len)
{
    //aligned to 1 page
    uint32_t waddr=file->data_addr+file->read_ptr;
    uint32_t waddr_end=waddr+len;
    waddr=(waddr+0x3FF)&(~0x3FF);
    waddr_end=waddr_end&(~0x3FF);
    while (waddr<waddr_end)
    {
        if (flash_sector_erase(waddr))
        {
            return -2;
        }
        waddr+=1024;
    }
    return 0;
}
/**
  * @brief  Write file in flash faster, no erase in write progress 
  *         `len` must be multiply of 256,
  *         file write ptr must be 256 aligned
  *         call `flash_file_prepare_write` first
  * @retval >0 ==> Writen bytes
  *         -2 ==> Write error
  */
int flash_file_fast_write(FlashFile*file,void*buf,uint32_t len)
{
    uint32_t wlen=0;
    while (wlen<len)
    {
        uint32_t waddr=file->data_addr+file->read_ptr;
        uint32_t wsize=256-(waddr&0xFF);
        if (len<wlen+wsize)
        {
            wsize=len-wlen;
        }
        if (flash_write(waddr,(uint8_t*)buf+wlen,wsize))
        {
            return -2;
        }
        wlen+=wsize;
        file->read_ptr+=wsize;
    }
    
    return wlen;
}
/**
  * @brief Open file in flash
  * @retval  0 ==> Success
  */
int flash_file_delete(FlashFile*file)
{
    uint32_t flash_index_addr = file->index * 64;
    uint32_t temp[2]={0,0};
    flash_write(flash_index_addr,temp,8);
    
    return 0;
}
