/**********************************************************************************
 *
 * @file    flash_shell.c
 * @brief   shell operate flash
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
 
/* Includes ------------------------------------------------------------------ */ 
#include <stdio.h>
#include <string.h>
#include "main.h"
#include "xmodem_frame.h"
#include "spi_flash.h"
#include "flash_util.h"
#include "shell_util.h"
#include "stdio_uart.h"

/* Private Macros ------------------------------------------------------------ */
#define FLASH_CAPACITY (8*1024*1024) //8MB

/* Private Variables --------------------------------------------------------- */
uint32_t flash_index_addr = 0;
uint32_t flash_write_num = 0;
uint32_t flash_data_addr = 0;
static char * temp = (char *)frame_env.data;

/* Public Variables ---------------------------------------------------------- */
extern boot_frame_env_t frame_env;
extern uart_handle_t g_uart_init;
extern crc_handle_t g_h_crc;
extern struct ShellEnv shell_env;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */


uint32_t calc_crc_ccitt(uint8_t *data, uint32_t initial, uint32_t len)
{
    CRC_RESET(&g_h_crc);
    return ald_crc_calculate(&g_h_crc, data, len);
}
void setup_crc_ccitt()
{
    /* Initialize CRC */
    g_h_crc.perh = CRC;
    g_h_crc.init.mode = CRC_MODE_CCITT;
    g_h_crc.init.seed = 0;
    g_h_crc.init.data_rev = DISABLE;
    g_h_crc.init.data_inv = DISABLE;
    g_h_crc.init.chs_rev = DISABLE;
    g_h_crc.init.chs_inv = DISABLE;

    g_h_crc.cal_cplt_cbk =  NULL;
    g_h_crc.err_cplt_cbk = NULL;
    ald_crc_init(&g_h_crc);
}

__STATIC_INLINE uint32_t LL_SYSTICK_IsActiveCounterFlag(void)
{
    return ((SysTick->CTRL & SysTick_CTRL_COUNTFLAG_Msk) == (SysTick_CTRL_COUNTFLAG_Msk));
}
uint64_t getCurrentMicros(void)
{
    /* Ensure COUNTFLAG is reset by reading SysTick control and status register */
    LL_SYSTICK_IsActiveCounterFlag();
    uint32_t m = ald_get_tick();
    const uint32_t tms = SysTick->LOAD + 1;
    __IO uint32_t u = tms - SysTick->VAL;
    if (LL_SYSTICK_IsActiveCounterFlag())
    {
        m = ald_get_tick();
        u = tms - SysTick->VAL;
    }
    return ((uint64_t)m) * 1000 + (u * 1000) / tms;
}

int download_cbk(void *usrdata, int idx, uint8_t *data, uint32_t crc)
{
    if ((flash_data_addr & 0xFFF) == 0)
    {
        flash_sector_erase(flash_data_addr);
    }
    for (uint32_t p = 0; p < 1024; p += 256)
    {

        flash_write(flash_data_addr, &data[p], 256);
        flash_data_addr += 256;
    }

    flash_write_num += 1024;
    return 0;
}

int send_cbk(void *usrdata, int idx, uint8_t *data, uint32_t crc)
{
    FlashFile*f=(FlashFile*)usrdata;
    return flash_file_read(f,data,1024);
}

int cmd_download(int argc, char **argv)
{
    uint32_t data_addr = 0;
    int idx = flash_find_free_index(&data_addr);
    uint32_t file_size_expect = 0;
    if (idx == -1)
    {
        printf("All index has been filled\r\n");
        return 0;
    }
    flash_index_addr = idx * 64;
    flash_data_addr = data_addr;
    flash_write_num = 0;

    printf("Prepare to download to flash\r\n");
    printf("File index=%d\r\n", idx);
    printf("File addr =%X\r\n", data_addr);

    /* Write index info */
    flash_write(flash_index_addr, &flash_data_addr, 4);

    //file name info
    if (argc >= 2)
    {
        uint32_t slen = strlen(argv[1]) + 1;
        if (slen > 56)
        {
            argv[1][55] = 0;
            slen = 56;
        }
        flash_write(flash_index_addr + 8, argv[1], slen);
    }
    else
    {
        uint32_t dummy = 0;
        flash_write(flash_index_addr + 8, &dummy, 4);
    }

    //file size info
    if (argc >= 3)
    {
        file_size_expect = get_uint(argv[2]);
    }

    setup_crc_ccitt();
    uart_enable_rx_irq(0);
    xm_frame_init(download_cbk, NULL);
    xm_download();
    uart_enable_rx_irq(1);

    if (file_size_expect < flash_write_num && flash_write_num - file_size_expect < 1024)
    {
        flash_write(flash_index_addr + 4, &file_size_expect, 4);
    }
    else
    {
        flash_write(flash_index_addr + 4, &flash_write_num, 4);
    }
    flash_wait_unbusy();

    printf("Success (%u bytes downloaded)\r\n", flash_write_num);
    return 0;
}


int cmd_getfile(int argc, char **argv)
{
    int idx = flash_file_find(argv[1]);
    FlashFile f;
    int ret=0;
    
    if (idx<0)
    {
        printf("File not found!\r\n");
        return -1;
    }
    flash_file_open(idx,&f);
    printf("Prepare to send\r\n");
    printf("File index=%d\r\n", idx);
    printf("File addr =%X\r\n", f.data_addr);
    printf("File size =%u\r\n", f.data_size);

    setup_crc_ccitt();
    uart_enable_rx_irq(0);
    xm_frame_init(send_cbk, &f);
    ret=xm_send();
    uart_enable_rx_irq(1);

    if (ret==0)
    {
        printf("Success\r\n");
    }
    else
    {
        printf("Error %d\r\n",ret);
    }
    return 0;
}
int cmd_id(int argc, char **argv)
{
    uint32_t id = flash_read_id();
    printf("Read Flash ID=%x\r\n", id);
    return 0;
}
int cmd_info(int argc, char **argv)
{
    printf("Reading flash info index\r\n");
    uint32_t p = 0;
    uint8_t buf[65];
    uint32_t data_ends = 4096;
    for (p = 0; p < 4096; p += 64)
    {
        flash_read(p, (char *)buf, 64);
        uint32_t faddr = ((uint32_t *)buf)[0];
        uint32_t fsize = ((uint32_t *)buf)[1];
        buf[64] = 0;
        if (faddr != 0xFFFFFFFF)
        {
            uint32_t fend = faddr + fsize;
            if (fend > data_ends)
            {
                data_ends = fend;
            }
            printf("[%02d]0x%08X (%u) \"%s\"\r\n", p / 64, faddr, fsize, buf + 8);
        }
    }
    printf("Total %u, %u used, %u free\r\n",FLASH_CAPACITY,data_ends,FLASH_CAPACITY-data_ends);
    return 0;
}
int cmd_erase(int argc, char **argv)
{
    uint32_t erase_beg = 0;
    uint32_t erase_end = 0;
    if(argc==2)
    {
        if (!strcmp(argv[1],"all"))
        {
            printf("Running chip erase, please waiting\r\n"); 
            printf("It will take 20-60 seconds\r\n");
            flash_chip_erase();
            flash_wait_unbusy();
            printf("Completed\r\n");
        }
    }
    else if (argc == 3)
    {
        erase_beg = get_uint(argv[1]) & (~0xFFF);
        erase_end = get_uint(argv[2]) & (~0xFFF);
        printf("Erasing %08X-%08X\r\n", erase_beg, erase_end);
        for (uint32_t p = erase_beg; p <= erase_end; p += 4096)
        {
            flash_sector_erase(p);
        }
    }
    else
    {
        printf("Invalid param\r\n");
    }
    return 0;
}
int cmd_init(int argc, char **argv)
{

    printf("Erasing sector 0 to init index\r\n");
    flash_sector_erase(0);
    return 0;
}
int cmd_read(int argc, char **argv)
{
    uint32_t read_beg = 0;
    uint32_t read_end = 0;
    if (argc < 2)
    {
        printf("CMD: read BEGIN [END]\r\n");
    }
    else
    {
        read_beg = get_uint(argv[1]);
        if (argc < 3)
            read_end = read_beg + 4;
        else
            read_end = get_uint(argv[2]);
        printf("Reading %08X-%08X\r\n", read_beg, read_end);

        for (uint32_t p = read_beg; p < read_end; p++)
        {
            if ((p - read_beg) % 16 == 0)
            {
                printf("\r\n");
            }

            uint8_t data;
            flash_read(p, (char *)&data, 1);
            printf("%02X ", data);
        }
        printf("\r\n");
    }
    return 0;
}
int cmd_crc(int argc, char **argv)
{
    int index = 0;
    if (argc < 2)
    {
        printf("USAGE: crc [INDEX]\r\n");
        return 0;
    }
    index = get_uint(argv[1]);
    printf("CRC [%d]=\r\n", index);
    return 0;
}
int cmd_cat(int argc, char **argv)
{
    int index = 0;
    if (argc < 2)
    {
        printf("USAGE: cat [INDEX/NAME]\r\n");
        return 0;
    }

    char *name = argv[1];
    if (name[0] == '/')
    {
        index = flash_file_find(argv[1] + 1);
    }
    else
    {
        if (name[0] >= '0' && name[0] <= '9')
        {
            index = get_uint(name);
        }
        else
        {
            index = flash_file_find(name);
        }
    }
    if (index < 0)
    {
        printf("File \"%s\" not found\r\n", name);
    }

    FlashFile file;
    int ret = 0;
    if ((ret = flash_file_open(index, &file)) < 0)
    {
        printf("Open fail %d\r\n", ret);
    }

    while ((ret = flash_file_read(&file, temp, 1024)) > 0)
    {
        uart_send_str(temp, ret, 1000);
    }
    if (ret < 0)
    {
        printf("Read error %d\r\n", ret);
        return -1;
    }
    return 0;
}

int cmd_test(int argc, char **argv)
{
    uint64_t tus;
    printf("====================\r\n");
    printf("Read test:\r\n");
    tus = getCurrentMicros();
    for (int i = 0; i < 32; i++)
    {
        flash_read(i * 1024, temp, 1024);
    }
    tus = getCurrentMicros() - tus;

    uint32_t ms, us;
    ms = tus / 1000;
    us = tus % 1000;
    printf("Used %d.%03d ms\r\n", ms, us);
    printf("Speed %d bytes/s\r\n", (32 * 1024) * 1000 / ms);
    
    int errors=0;
    FlashFile f;
    
    printf("====================\r\n");
    printf("Write test:\r\n");
    flash_file_new(&f,"test",0);
    for (int i=0;i<1024;i++)
    {
        temp[i]=i&0xFF;
    } 
    tus = getCurrentMicros();
    for (int i=0;i<64;i++)
    {
        if (flash_file_write(&f,temp,1024)!=1024)
        {
            printf("Write error\r\n");
            break;
        }
        if (i%4==3)
        {
            printf(".");
        }
    }
    flash_file_close(&f);
    printf("\r\n");
    tus = getCurrentMicros() - tus;
    ms = tus / 1000;
    us = tus % 1000;
    printf("Used %d.%03d ms\r\n", ms, us);
    printf("Speed %d bytes/s\r\n", (64 * 1024) * 1000 / ms);
    
    printf("Checking...\r\n");
    memset(&f,0,sizeof(f));
    if (flash_file_open(flash_file_find("test"),&f)<0)
    {
        printf("Open test file failed\r\n");
    }
    printf("TestFile:Index: %d\r\n",f.index);
    printf("         Addr : %x\r\n",f.data_addr);
    printf("         Size : %u\r\n",f.data_size);
    for (int i=0;i<64;i++)
    {
        if (flash_file_read(&f,temp,1024)!=1024)
        {
            printf("Read error!\r\n");
            break;
        }
        for (int j=0;j<1024;j++)
        {
            if (temp[j]!=(temp[j]&0xFF))
            {
                errors++;
            }
        }
    }
    printf("%d errors\r\n",errors);
    
    flash_file_delete(&f);
    
    
    printf("====================\r\n");
    printf("FastWrite test:\r\n");
    flash_file_new(&f,"testfast",0);
    for (int i=0;i<1024;i++)
    {
        temp[i]=i&0xFF;
    }
    printf("Erasing...\r\n");
    flash_file_prepare_write(&f,64*1024);
    printf("Writing...\r\n");
    tus = getCurrentMicros();
    for (int i=0;i<64;i++)
    {
        if (flash_file_fast_write(&f,temp,1024)!=1024)
        {
            printf("Write error\r\n");
            break;
        }
        if (i%4==3)
        {
            printf(".");
        }
    }
    flash_file_close(&f);
    printf("\r\n");
    tus = getCurrentMicros() - tus;
    ms = tus / 1000;
    us = tus % 1000;
    printf("Used %d.%03d ms\r\n", ms, us);
    printf("Speed %d bytes/s\r\n", (64 * 1024) * 1000 / ms);
    
    printf("Checking...\r\n");
    memset(&f,0,sizeof(f));
    if (flash_file_open(flash_file_find("testfast"),&f)<0)
    {
        printf("Open test file failed\r\n");
    }
    printf("TestFile:Index: %d\r\n",f.index);
    printf("         Addr : %x\r\n",f.data_addr);
    printf("         Size : %u\r\n",f.data_size);
    for (int i=0;i<64;i++)
    {
        if (flash_file_read(&f,temp,1024)!=1024)
        {
            printf("Read error!\r\n");
            break;
        }
        for (int j=0;j<1024;j++)
        {
            if (temp[j]!=(temp[j]&0xFF))
            {
                errors++;
            }
        }
    }
    printf("%d errors\r\n",errors);
    
    flash_file_delete(&f);
    return 0;
}

int cmd_file(int argc, char **argv)
{
    static FlashFile f;
    int ret=0;
    if (argc<2)
    {
        return 0;
    }
    if (!strcmp(argv[1],"create"))
    {
        uint32_t len=0;
        if (argc>=4)
        {
            len=get_uint(argv[3]);
        }
        ret=flash_file_new(&f,argv[2],len);
    }
    else if (!strcmp(argv[1],"open"))
    {
        ret=flash_file_open(flash_file_find("test"),&f);
    }
    else if (!strcmp(argv[1],"close"))
    {
        ret=flash_file_close(&f);
        memset(&f,0,sizeof(f));
    }
    else if (!strcmp(argv[1],"write"))
    {
        ret=flash_file_write(&f,(void*)argv[2],strlen(argv[2]));
    }
    else if (!strcmp(argv[1],"read"))
    {
        uint32_t len=get_uint(argv[2]);
        ret=flash_file_read(&f,(void*)temp,len);
        temp[len]=0;
        uart_send_str(temp,len,1000);
    }
    else if (!strcmp(argv[1],"seek"))
    {
        f.read_ptr=get_uint(argv[2]);
    }
    else if (!strcmp(argv[1],"index"))
    {
        ret=flash_file_find(argv[2]);
    }
    printf("Result=%d\r\n",ret);
    return 0;
}
int cmd_exit(int argc, char **argv)
{
    shell_env.flag|=SHELL_EXIT;
    return 0;
}

struct CMD_FUNC flash_cmd_table[] =
{
    {"init", cmd_init},
    {"id", cmd_id},
    {"info", cmd_info},
    {"read", cmd_read},
    {"erase", cmd_erase},
    {"down", cmd_download},
    {"crc", cmd_crc},
    {"cat", cmd_cat},
    {"test", cmd_test},
    {"quit", cmd_exit},  
    {"exit", cmd_exit},
    {"file",cmd_file},
    {"getfile",cmd_getfile},
    {NULL, NULL}
};
struct ShellEnv shell_env={
    .buf=(char *)frame_env.data,
    .buf_len=1024,
    .cmd_table=flash_cmd_table,
};
void enter_flash_shell()
{
    if (ald_gpio_read_pin(KEY_CENTER_PORT, KEY_CENTER_PIN) == 0)
    {
        printf("Enter flash operator mode!\r\n");
        enter_shell_loop(&shell_env);
    }
}
