/**********************************************************************************
 *
 * @file    dac_main.c
 * @brief   dac init and output main loop
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
 
/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include <stdio.h> 
#include "main.h"
#include "audio_read.h"

/* Private Macros ------------------------------------------------------------ */
/* DAC dma channel(only used in dac mode) */
#define DMA_DAC_CHANNEL 0 

/* Private Variables --------------------------------------------------------- */
struct AudioDMAHandler
{
    dma_config_t dmacfg;
    int16_t *buf;
    uint32_t len;
};
static timer_handle_t h_tim;
static dac_channel_config_t channel_config;  
static struct AudioDMAHandler audio_dma[2];

/* Public Variables ---------------------------------------------------------- */
dac_handle_t g_h_dac;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */
void dma_cplt_cbk(void *arg);

/* Private Function ---------------------------------------------------------- */


/**
  * @brief  DAC pin init.
  * @retval None.
  */
void dac_gpio_init(void)
{
    gpio_init_t x;
    memset(&x, 0, sizeof(gpio_init_t));

    x.mode = GPIO_MODE_OUTPUT;
    x.func = GPIO_FUNC_0;
    ald_gpio_init(DAC0_OUT_PORT, DAC0_OUT_PIN, &x);
    ald_gpio_init(DAC1_OUT_PORT, DAC1_OUT_PIN, &x);
}

/**
  * @brief  Initializate DMA chanel0 and channel1.
  * @retval None
  */
void timer_init(void)
{
    /* Initialize AD16C4T0 */
    h_tim.perh           = AD16C4T0;
    h_tim.init.prescaler = 45 - 1;
    h_tim.init.mode      = TIMER_CNT_MODE_UP;
    h_tim.init.period    = 200 - 1;
    h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    h_tim.capture_cbk    = NULL;
    ald_timer_base_init(&h_tim);
    ald_timer_base_start(&h_tim);
}

/**
  * @brief  DAC DMA complete
  * @retval None.
  */
void dma_pingpong_complete(void *arg)
{
    dma_descriptor_t *desc;
    struct AudioDMAHandler *hdma;
    desc = (dma_descriptor_t *)(DMA0->CTRLBASE) + DMA_DAC_CHANNEL;
    if (desc->ctrl.cycle_ctrl == 0)
    {
        //primary
        hdma = &audio_dma[0];
    }
    else
    {
        desc = (dma_descriptor_t *)(DMA0->ALTCTRLBASE) + DMA_DAC_CHANNEL;
        hdma = &audio_dma[1];
    }
    #if USE_INTERNAL_FLASH
    prepare_audio(hdma->buf);
    #endif
    ald_dma_config_ping_pong(DMA0, &hdma->dmacfg, 0, dma_pingpong_complete);
}

/**
  * @brief  Init DMA handler
  * @retval None.
  */
void init_hdma(int idx)
{
    struct AudioDMAHandler *hdma = &audio_dma[idx];
    hdma->buf = &sample_buffer[idx * 1024];
    hdma->len = 1024;
    #if USE_INTERNAL_FLASH
    prepare_audio(hdma->buf);
    #endif

    memset(&hdma->dmacfg, 0x0, sizeof(dma_config_t));
    hdma->dmacfg.src        = hdma->buf;
    hdma->dmacfg.dst        = (void *)&DAC0->CH0DATA;
    hdma->dmacfg.size       = hdma->len;
    hdma->dmacfg.R_power    = DMA_R_POWER_1;
    hdma->dmacfg.data_width = DMA_DATA_SIZE_HALFWORD;
    hdma->dmacfg.src_inc    = DMA_DATA_INC_HALFWORD;;
    hdma->dmacfg.dst_inc    = DMA_DATA_INC_NONE;
    hdma->dmacfg.primary    = (idx == 0) ? ENABLE : DISABLE;
    hdma->dmacfg.burst      = ENABLE;
    hdma->dmacfg.high_prio  = DISABLE;
    hdma->dmacfg.interrupt  = ENABLE;
    hdma->dmacfg.channel    = DMA_DAC_CHANNEL;
    hdma->dmacfg.msel       = DMA_MSEL_TIMER0;
    hdma->dmacfg.msigsel    = DMA_MSIGSEL_TIMER_UPDATE;

    ald_dma_config_ping_pong(DMA0, &hdma->dmacfg, (idx == 0 ? 1 : 0), dma_pingpong_complete);
}

/**
  * @brief  Start DAC audio play
  * @retval None.
  */
void start_dac_audio()
{
    
    init_hdma(0);
    init_hdma(1);

    timer_init();
    AD16C4T0->DMAEN |= (1 << 0);
}
/**
  * @brief  Init DAC
  * @retval Status.
  */
void init_dac()
{
    /* DAC pin Initialize */
    dac_gpio_init();

    memset(&g_h_dac, 0, sizeof(dac_handle_t));
    memset(&channel_config, 0, sizeof(dac_channel_config_t));

    /* Initialize DAC */
    g_h_dac.perh            = DAC0;
    g_h_dac.init.conv_mode  = DAC_CONV_MODE_CONTINUOUS;
    g_h_dac.init.out_mode   = DAC_OUTPUT_PIN;
    g_h_dac.init.refresh    = DAC_REFRESH_8;
    g_h_dac.init.div        = DAC_PRES_DIV_128;
    g_h_dac.init.ch0_reset  = DISABLE;
    g_h_dac.init.o_ctrl_pis = DISABLE;
    g_h_dac.init.sine       = DISABLE;
    g_h_dac.init.diff       = DISABLE;
    g_h_dac.init.n_ref      = DAC_NEG_REF_VSS;
    g_h_dac.init.p_ref      = DAC_POS_REF_VDD;
    g_h_dac.cbk             = NULL;
    ald_dac_init(&g_h_dac);

    /* Config dac channel 0 */
    channel_config.enable     = ENABLE;
    channel_config.trigger    = DAC_TRIGGER_BY_DATA;
    channel_config.refresh_en = DISABLE;
    ald_dac_channel_config(&g_h_dac, &channel_config, DAC_CHANNEL_0);


} 
void start_dac()
{
    init_dac();
    start_dac_audio();
}
