/**********************************************************************************
 *
 * @file    audio_read.c
 * @brief   read audio data from spi flash or internal flash
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
 
/* Includes ------------------------------------------------------------------ */
#include <stdio.h>
#include <string.h>
#include "audio_read.h"
#include "flash_util.h"
#include "main.h"
#include "adpcm.h"

/* Private Macros ------------------------------------------------------------ */
#define U2C4(X) (X)&0xFF,(X>>8)&0xFF,(X>>16)&0xFF,(X>>24)&0xFF
#define AUDIO_BUFFER 1024
/* Private Variables --------------------------------------------------------- */
int wav_header_size=44;
static int read_len=0;
static int recv_len=0;
static int16_t* read_buf=NULL;
static uint8_t g_use_adpcm=0;

/* Public Variables ---------------------------------------------------------- */
uint32_t wav_ptr = 0;
int16_t sample_buffer[AUDIO_BUFFER * 2];
FlashFile wav_file;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */


void print_wav_info(wave_formattypedef_t *wav)
{
    printf("Chunk ID     = %X,%c%c%c%c\r\n", wav->ChunkID, U2C4(wav->ChunkID));
    printf("FileSize     = %d\r\n", wav->FileSize);
    printf("FileFormat   = %d,%c%c%c%c\r\n", wav->FileFormat, U2C4(wav->FileFormat));
    printf("SubChunk1ID  = %X,%c%c%c%c\r\n", wav->SubChunk1ID, U2C4(wav->SubChunk1ID));
    printf("SubChunk1Size= %d\r\n", wav->SubChunk1Size);
    printf("AudioFormat  = %d\r\n", wav->AudioFormat);
    printf("NbrChannels  = %d\r\n", wav->NbrChannels);
    printf("SampleRate   = %d\r\n", wav->SampleRate);
    printf("ByteRate     = %d\r\n", wav->ByteRate);
    printf("BlockAlign   = %d\r\n", wav->BlockAlign);
    printf("BitPerSample = %d\r\n", wav->BitPerSample);
    printf("SubChunk2ID  = %X,%c%c%c%c\r\n", wav->SubChunk2ID, U2C4(wav->SubChunk2ID));
    printf("SubChunk2Size= %d\r\n", wav->SubChunk2Size);
}

#if USE_INTERNAL_FLASH
void prepare_audio(int16_t *buf)
{
    if (g_use_adpcm)
    {
        uint8_t *audio_p=(uint8_t *)&g_sample_wav_data[wav_ptr];
        int rlen=512;
        /* 4-bit ADPCM read 512 bytes (1024 samples)*/
        if (wav_ptr + AUDIO_BUFFER/2 >= g_sample_wav_data_len)
        {
            rlen=(g_sample_wav_data_len-wav_ptr)/2;
            wav_ptr = 44;
        }
        else
        {
            wav_ptr += AUDIO_BUFFER/2;
        }
        
        for (int i=0;i<rlen;i++)
        {
            buf[i*2]=ADPCM_Decode((audio_p[i])&0xF);
            buf[i*2+1]=ADPCM_Decode((audio_p[i]>>4)&0xF);
        }
        for (int i=rlen;i<AUDIO_BUFFER/2;i++)
        {
            buf[i*2]=0;
            buf[i*2+1]=0;
        }
    }
    else
    {
        int16_t *audio_p=(int16_t *)&g_sample_wav_data[wav_ptr];
        int rlen=1024;
        /* 16-bit RAW read 2048 bytes (1024 samples)*/
        if (wav_ptr + AUDIO_BUFFER * 2 >= g_sample_wav_data_len)
        {
            rlen=(g_sample_wav_data_len-wav_ptr)/2;
            wav_ptr = 44;
        }
        else
        {
            wav_ptr += AUDIO_BUFFER * 2;
        }
        for (int i = 0; i < rlen; i++)
        {
            buf[i] = audio_p[i];
        }
        for (int i=rlen;i<AUDIO_BUFFER;i++)
        {
            buf[i]=0;
        }
    }
}
#endif
void set_adpcm(uint8_t enabled)
{
    g_use_adpcm=enabled;
}
int setup_flash_wav(const char*name)
{
    int ret=flash_file_open(flash_file_find(name),&wav_file);
    if (ret<0)
    {
        memset(&wav_file,0,sizeof(wav_file));
        printf("Read \"%s\" failed:%d \r\n",name,ret);
        return ret;
    }
    
    wave_formattypedef_t wav_header;
    
    /* read wav header */
    wav_header_size=44;
    flash_file_read(&wav_file,&wav_header,44); /* at least 44 bytes */
    print_wav_info(&wav_header);
    if (wav_header.BitPerSample==4)
    {
        ADPCM_Reset();
        g_use_adpcm=1;
    }
    else
    {
        g_use_adpcm=0;
    }
    
    init_sample_timer(wav_header.SampleRate);
    if (wav_header.SubChunk2ID!=WAVF_data)
    {
        /* This is not data find next subchunk*/
        int subchunk=3;
        wav_header_size+=wav_header.SubChunk2Size+8;
        while (1)
        {
            uint32_t t[2];
            wav_file.read_ptr=wav_header_size-8;
            flash_file_read(&wav_file,t,8); 
            printf("SubChunk%dID  =%X,%c%c%c%c\r\n",subchunk,t[0],U2C4(t[0]));
            printf("SubChunk%dSize= %d\r\n",subchunk, t[1]);
            if (t[0]==WAVF_data)
            {
                break;
            }
            else
            {
                wav_header_size+=t[1];
            }
        }
    }
    
    return wav_header.SampleRate;
}

void read_audio_cplt()
{
    if (recv_len<1024)
    {
        memset(read_buf+recv_len,0,1024-recv_len);
        wav_file.read_ptr=wav_header_size;
    }
    
    for (int i = 0; i < 512; i++)
    {
        //sacle to 0-255
        int val=read_buf[i];
        read_buf[i] = (32768 + val) / 256;
    }
    if (read_len>1024)
    {    
        read_len-=1024;
        read_buf=(int16_t*)((uint8_t*)read_buf+1024);
        //recv_len=flash_file_read_dma(&wav_file,read_buf,1024,read_audio_cplt);
    }
    else
    {
        read_len=0;
        read_buf=NULL;
    }

}

void read_audio_data(int16_t *buf)
{
    if (read_len||read_buf)
    {
        printf("WARNING: read overlapped!\r\n");
        return;
    }
    if (wav_file.data_size==0)
    {
        memset((uint8_t*)read_buf+recv_len,0,read_len-recv_len);
        return ;
    }
    if (g_use_adpcm)
    {
        /* read 512 bytes (1024 samples) */
        uint8_t*tmp=((uint8_t*)buf)+2048-512;
        read_buf=buf;
        read_len=512;
        recv_len=flash_file_read(&wav_file,tmp,read_len);
        if (recv_len<read_len)
        {
            memset(tmp+recv_len,0,read_len-recv_len);
            wav_file.read_ptr=wav_header_size;
        }
        for (int i=0;i<read_len;i++)
        {
            read_buf[i*2]=ADPCM_Decode(tmp[i]&0xF);
            read_buf[i*2+1]=ADPCM_Decode(tmp[i]>>4);
        }
        read_buf=NULL;
        read_len=0;
    }
    else
    {
        /* read 2048 bytes (1024 samples) */
        read_buf=buf;
        read_len=1024*2;
        recv_len=flash_file_read(&wav_file,buf,read_len);
        if (recv_len<read_len)
        {
            memset((uint8_t*)read_buf+recv_len,0,read_len-recv_len);
            wav_file.read_ptr=wav_header_size;
        }
        read_buf=NULL;
        read_len=0;
    }
}
