/**********************************************************************************
 *
 * @file    pwm_main.c
 * @brief   pwm output main loop
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include <stdio.h>
#include "main.h"
#include "audio_read.h"
#include "shell_util.h"
#include "flash_util.h"
#include "stdio_uart.h"

/* Private Macros ------------------------------------------------------------ */

#define  SAMPLE_RATE    8000
#define  PWM_DAC_BITS   10                             /* Target PWM-DAC bits */
#define  PWM_DAC_VALUE  (1<<PWM_DAC_BITS)
#define  PERIOD_VALUE   (uint32_t)(PWM_DAC_VALUE - 1) /* Period Value  */
#define  PULSE1_VALUE   (uint32_t)(PWM_DAC_VALUE/2)   /* Capture Compare 1 Value  */
#define  PULSE2_VALUE   (uint32_t)(PWM_DAC_VALUE/4)   /* Capture Compare 2 Value  */

/* Private Variables --------------------------------------------------------- */

/* PWM output timer */
static timer_handle_t pwm_timer;         /* GP16C4T0 */
static timer_clock_config_t pwm_tim_clock;
static timer_oc_init_t pwm_tim_ocinit;

/* Audio sample timer */
timer_handle_t sampler_timer;            /* AD16C4T1 */
static timer_clock_config_t smp_tim_clock;

/* PWM/DAC output */
static uint32_t sample_ptr=0;
static volatile int should_feed_data=0;
static volatile int should_save_data=0;
static int output_test_mode=0;
static int cur_sample_rate=0;

/* ADC sample */
static uint16_t adc_buf[2048]={0};
static int adc_writer=0;
static int adc_dac_loopback=0;
static FlashFile save_file;

/* Public Variables ---------------------------------------------------------- */
extern struct ShellEnv shell_env;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */


/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void tim_pin_init(void)
{
    gpio_init_t x;

    /* Initialize tim0 ch1 pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.func = GPIO_FUNC_2;

    ald_gpio_init(GP16C4T0_CHANNEL_PORT, GP16C4T0_CHANNEL1_PIN, &x);
    ald_gpio_init(GP16C4T0_CHANNEL_PORT, GP16C4T0_CHANNEL2_PIN, &x);
}
/**
  * @brief  ald timer period elapsed callback
  * @param  arg: Pointer to timer_handle_t structure.
  * @retval None.
  */
void timer_period_elapsed_callback(struct timer_handle_s *arg)
{
    uint32_t val;
    
    adc_buf[adc_writer]=(ADC0->NCHDR)<<4;
    
    if (adc_dac_loopback)
    {
        val=(adc_buf[adc_writer])>>1;
    }
    else
    {
        val=sample_buffer[sample_ptr]+32768;
        sample_ptr++;
        if (sample_ptr==1024)
        {
            should_feed_data=1;
        }
        if (sample_ptr==2048)
        {
            should_feed_data=2;
            sample_ptr=0;
        }
    }
    
    DAC0->CH0DATA=(val>>4);
    WRITE_REG(pwm_timer.perh->CCVAL1, val>>(16-PWM_DAC_BITS));
    
    adc_writer++;
    if (adc_writer==1024)
    {
        should_save_data=1;
    }
    if (adc_writer==2048)
    {
        should_save_data=2;
        adc_writer=0;
    }
}

void init_sample_timer(uint32_t sample_rate)
{
    uint32_t prescaler;
    uint32_t period;
    if (sample_rate==cur_sample_rate)
    {
        /* Same, nothing to do */
        return;
    }
    cur_sample_rate=sample_rate;
    switch (sample_rate)
    {
        case 16000:
            prescaler=45;
            period=100;
            break;
        case 12000:
            prescaler=60;
            period=100;
            break;
        case 8000: 
        default:
            prescaler=45;
            period=200;
            break;
    }
    printf("Init sample rate=%d (prescaler=%d,period=%d)\r\n",sample_rate,prescaler,period);
    if (sampler_timer.state!=TIMER_STATE_RESET)
    {
        ald_timer_base_stop_by_it(&sampler_timer);
        ald_timer_base_reset(&sampler_timer);
    }
    sampler_timer.perh = AD16C4T1;
    sampler_timer.init.prescaler    = prescaler - 1;             
    sampler_timer.init.mode         = TIMER_CNT_MODE_UP;  /* count up */
    sampler_timer.init.period       = period - 1;           
    sampler_timer.init.clk_div      = TIMER_CLOCK_DIV1;   /* working clock of dead time and filter */
    sampler_timer.init.re_cnt       = 0;
    sampler_timer.period_elapse_cbk = timer_period_elapsed_callback;  /* updata period callback function */
    ald_timer_base_init(&sampler_timer);

    /* Initialize clock source */
    smp_tim_clock.source = TIMER_SRC_INTER;   /**< internal clock sources */
    ald_timer_config_clock_source(&sampler_timer, &smp_tim_clock);

    ald_mcu_irq_config(AD16C4T1_UP_IRQn, 0, 0, ENABLE);/* Enable AD16C4T1 interrupt */
    ald_timer_base_start_by_it(&sampler_timer);       /* Start UPDATE interrupt by interrupt */
}

void setup_pwm()
{
    /* Initialize GP16C4T0 */
    pwm_timer.perh           = GP16C4T0;
    pwm_timer.init.prescaler = 1;
    pwm_timer.init.mode      = TIMER_CNT_MODE_UP;
    pwm_timer.init.period    = (1<<PWM_DAC_BITS)-1;
    pwm_timer.init.clk_div   = TIMER_CLOCK_DIV1;
    pwm_timer.init.re_cnt    = 0;
    ald_timer_pwm_init(&pwm_timer);

    /* Initialize clock source */
    pwm_tim_clock.source = TIMER_SRC_INTER;
    ald_timer_config_clock_source(&pwm_timer, &pwm_tim_clock);

    /* Common configuration for all channels */
    pwm_tim_ocinit.oc_mode      = TIMER_OC_MODE_PWM1;
    pwm_tim_ocinit.oc_polarity  = TIMER_OC_POLARITY_HIGH;
    pwm_tim_ocinit.oc_fast_en   = ENABLE;
    pwm_tim_ocinit.ocn_polarity = TIMER_OCN_POLARITY_HIGH;
    pwm_tim_ocinit.ocn_idle     = TIMER_OCN_IDLE_RESET;
    pwm_tim_ocinit.oc_idle      = TIMER_OC_IDLE_RESET;

    /* Set the pulse value for channel 1 */
    pwm_tim_ocinit.pulse = PULSE1_VALUE;
    ald_timer_oc_config_channel(&pwm_timer, &pwm_tim_ocinit, TIMER_CHANNEL_1);
    MODIFY_REG(pwm_timer.perh->CHMR1, TIMER_CHMR1_CH1OPREN_MSK, 1 << TIMER_CHMR1_CH1OPREN_POS);
    /* Set the pulse value for channel 2 */
    pwm_tim_ocinit.pulse = PULSE2_VALUE;
    ald_timer_oc_config_channel(&pwm_timer, &pwm_tim_ocinit, TIMER_CHANNEL_2);

    /* Start input pwm from tim0 channel 1 */
    ald_timer_oc_start(&pwm_timer, TIMER_CHANNEL_1);
    /* Start input pwm from tim0 channel 2 */
    ald_timer_oc_start(&pwm_timer, TIMER_CHANNEL_2);
}
void init_pwm()
{
    /* Initialize pin */
    tim_pin_init();
    
    setup_pwm();
}

static int cmd_play(int argc,char**argv)
{
    if (argc<2)
        return 0;
    
    int smp=setup_flash_wav(argv[1]);
    if (smp>0&&smp!=cur_sample_rate)
    {
        ald_timer_base_stop_by_it(&sampler_timer);
        init_sample_timer(smp);
    }
    return 0;
}
static int cmd_list(int argc,char**argv)
{
    FlashFileIter fi;
    fi.index=-1;
    printf("|     Name     |  Size  |\r\n");
    while (flash_file_list(&fi)==0)
    {
        printf("|%14s|%8u|\r\n",fi.file.name,fi.file.fsize);
    }
    return 0;
}

static int cmd_output_test(int argc,char**argv)
{
    int set=0;
    if (argc<2)
    {
        printf("Output test mode = %d\r\n",output_test_mode);
        return 0;
    }
    else
    {
        if (argv[1][0]=='Y'||argv[1][0]=='y'||argv[1][0]=='1')
        {
            set=1;
        }
        else if (argv[1][0]=='N'||argv[1][0]=='n'||argv[1][0]=='0')
        {
            set=0;
        }
        else
        {
            printf("Invalid param: %s\r\n",argv[1]);
            return 0;
        }
    }
    if (set==output_test_mode)
    {
        return 0;
    }
    if (set)
    {
        printf("Output test mode enabled\r\n");
        ald_timer_base_stop_by_it(&sampler_timer);
    }
    else
    {
        printf("Output test mode disabled\r\n");
        ald_timer_base_start_by_it(&sampler_timer);
    }
    output_test_mode=set;
    return 0;
}

static int cmd_sample_rate(int argc,char**argv)
{
    if (argc<2)
    {
        return 0;
    }
    int new_rate=(int)get_uint(argv[1]);
    if (cur_sample_rate!=new_rate)
    {
        init_sample_timer(new_rate);
    }
    return 0;
}


static int cmd_save_adc(int argc,char**argv)
{
    if (argc<2)
    {
        return 0;
    }
    if (!strcmp(argv[1],"end"))
    {
        uint32_t wavlen=save_file.read_ptr;
        wave_formattypedef_t*wav_header=(wave_formattypedef_t*)shell_env.buf;
        wav_header->ChunkID=WAVF_RIFF;
        wav_header->FileSize=wavlen; 
        wav_header->FileFormat=WAVF_WAVE;
        wav_header->SubChunk1ID=WAVF_fmt;
        wav_header->SubChunk1Size=16;
        wav_header->AudioFormat=1;
        wav_header->NbrChannels=1;
        wav_header->SampleRate=16000;
        wav_header->ByteRate=32000;
        wav_header->BlockAlign=2;
        wav_header->BitPerSample=16;
        wav_header->SubChunk2ID=WAVF_data;
        wav_header->SubChunk2Size=wavlen-44;
        
        /*  Write wav file header */
        save_file.read_ptr=0;
        flash_file_fast_write(&save_file,shell_env.buf,44);
        save_file.read_ptr=wavlen;
        flash_file_close(&save_file);
        memset(&save_file,0,sizeof(save_file));
    }
    else
    {
        flash_file_new(&save_file,argv[1],0);
        save_file.read_ptr=44; //skip wav file header
    }
    return 0;
}
static int cmd_adc_loopback(int argc,char**argv)
{
    if (argc<2)
    {
        printf("adc_loopback = %d\r\n",adc_dac_loopback);
    }
    else
    {
        if (argv[1][0]=='Y'||argv[1][0]=='y'||argv[1][0]=='1')
        {
            adc_dac_loopback=1;
        }
        else if (argv[1][0]=='N'||argv[1][0]=='n'||argv[1][0]=='0')
        {
            adc_dac_loopback=0;
        }
        printf("ADC-Loopback set to %d\r\n", adc_dac_loopback);
    }
    return 0;
}

struct CMD_FUNC play_cmds[]={
    {"play",cmd_play},
    {"list",cmd_list},
    {"output_test",cmd_output_test},
    {"sample_rate",cmd_sample_rate},
    {"save_adc",cmd_save_adc},
    {"adc_loopback",cmd_adc_loopback},
    {NULL,NULL}
};

void start_audio()
{
    int keys[5]={0};
    int val=PWM_DAC_VALUE/2;
    /* initial read key states */
    for (int i=0;i<5;i++)
    {
        test_key(i,&keys[i]);
    }
    
    init_pwm();
    init_sample_timer(SAMPLE_RATE);
    memset(&save_file,0,sizeof(save_file));
    
    #if USE_SPI_FLASH
    /* Read wav file in flash */
    setup_flash_wav("t3.wav");
    #else
    /* Set ADPCM if necessary */
    //set_adpcm(1);
    #endif
    
    /* Init shell, data buffer shared with flash shell(xmodem buffer) */
    shell_env.buf_p=0;
    shell_env.flag=0;
    shell_env.cmd_table= play_cmds;
    uart_enable_rx_irq(1);
    while (1)
    {
        char ch=uart_peak_ch();;
        /* Receive from uart */
        if (ch)
        {
            shell_input(ch,&shell_env);
        }
        
        /* Need to read audio data */
        if (should_feed_data)
        {
            int16_t*p=sample_buffer+(should_feed_data-1)*1024;
            should_feed_data=0;
            
            if (output_test_mode)
            {
                memset(p,0,1024*2);
            }
            else
            {
                /* Read wav data */
                #if USE_SPI_FLASH
                read_audio_data(p);
                #else
                prepare_audio(p);
                #endif              
            }
        }
        
        if (should_save_data)
        {
            uint16_t*p=adc_buf+(should_save_data-1)*1024;
            should_save_data=0;
            if (save_file.data_addr!=0)
            {
                flash_file_fast_write(&save_file,p,1024*2);
            }
        }
        
        /* Output test mode */
        if (output_test_mode)
        {
            if (test_key(0,&keys[0])==1)
            {
                val=PWM_DAC_VALUE/2;
            }
            else if (test_key(1,&keys[1])==1)
            {
                if (val<PWM_DAC_VALUE-1)
                {
                    val++;
                }
            }
            else if (test_key(2,&keys[2])==1)
            {
                if (val>0)
                {
                    val--;
                }
            }
            else if (test_key(3,&keys[3])==1)
            {
                if (val>10)
                {
                    val-=10;
                }
            }
            else if (test_key(4,&keys[4])==1)
            { 
                if (val<PWM_DAC_VALUE-10)
                {
                    val+=10;
                }
            }
            else
            {
                continue;
            }
            DAC0->CH0DATA=val<<4;
            WRITE_REG(pwm_timer.perh->CCVAL1, val);
            printf("Val=%d\r\n",val);
        }
    }
    //uart_enable_rx_irq(0); /* this is unreachable */
}
