/**********************************************************************************
 *
 * @file    micro_sd_diskio.c
 * @brief   IO defination of spi flash
 *
 * @date    07 Dec 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          07 Dec 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "micro_sd_diskio.h"

/* Private Macros ------------------------------------------------------------ */

/* Private variables ---------------------------------------------------------*/

extern uint32_t sd_size;
/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS micro_sd_init(void);
DSTATUS micro_sd_stat(BYTE, void *);
DRESULT micro_sd_read(BYTE *, DWORD, UINT);
#if FF_FS_READONLY == 0
    DRESULT micro_sd_write(const BYTE *, DWORD, UINT);
#endif /* FF_FS_READONLY == 0 */
DRESULT micro_sd_ioctl(BYTE, void *);

/* Public Variables ---------------------------------------------------------- */
const gen_pdrv_io_t micro_sd_drv =
{
    micro_sd_init,
    micro_sd_stat,
    micro_sd_read,

#if FF_FS_READONLY == 0
    micro_sd_write,
#endif /* FF_FS_READONLY == 0 */
    micro_sd_ioctl,
};

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS micro_sd_init(void)
{
    /* Configure the NAND flash device */
    if (ll_micro_sd_init())
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS micro_sd_stat(BYTE parm_num, void *param)
{
    return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT micro_sd_read(BYTE *buff, DWORD sector, UINT count)
{
    if (ll_micro_sd_sector_read(buff, (uint32_t)(sector), count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if FF_FS_READONLY == 0
DRESULT micro_sd_write(const BYTE *buff, DWORD sector, UINT count)
{
    if (ll_micro_sd_sector_write(buff, sector, count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}
#endif /* FF_FS_READONLY == 0 */

/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT micro_sd_ioctl(BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = sd_size * BLK_SIZE;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = SEC_SIZE;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = BLK_SIZE;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            if (! ll_micro_sd_sector_erase(*(DWORD *)buff, *((DWORD *)buff + 1)))
                res = RES_OK;

            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
